# Functions for calculating coefficients ---------------------------------------------
# linking mean best number and Reynolds number, see CRYOWRF paper draft of Sharma et al.
# Used in sublimation function below

a_re = function(X,b){
  del_0 = 5.83
  c0 = 0.6
  c1 = 0.1519
  
  (((del_0^2.0)/4.0) * ( sqrt(1+c1*sqrt(X)) - 1.0 )^(2.0) ) / (X^b)
}


b_re = function(X){
  c1 = 0.1519
  
  denom =  ( ( sqrt(1+c1*sqrt(X)) - 1 ) * sqrt(1+c1 * sqrt(X)) )
  
  if(denom >= 1e-12){
    b = 0.5 * c1 * sqrt(X) / ( ( sqrt(1+c1*sqrt(X)) - 1 ) * sqrt(1+c1 * sqrt(X)) )
  } else{
    b = 1.0
  }
  return(b)
}


# Function for empirical linear parametrization of change of particle temperature -------------
# in lowest 0.3 m of the atmosphere. Can be used to compute the transient_term argument in the function below.

## Input:
# z           height (m)
# delta.T     temperature difference (K) of T(z=7.5mm) minus T(z=0)
# RH          relative humidity (%) at z=7.5mm
## Output:    intercept and slope define the change of particle temperature as a linear function of particle diameter 

coef_dTdt = function(z, delta.T, RH){
  # transform some input variables
  z.trans     = log(z)
  abs.delta.T = abs(delta.T)
  RH.trans    = sign(delta.T) * ( RH - 100 )
  # coefficients of linear function # coefficients can be derived from the equations in the paper by incorporating the subtraction of the mean and division by the standard deviation in the coefficients
  list(intercept = sign(delta.T) * (-0.171353 + 
                                    -0.060939 * z.trans +
                                     0.003081 * abs.delta.T +
                                     0.057531 * RH.trans +
                                    -0.197510 * z.trans * abs.delta.T ),
      slope      = sign(delta.T) * ( 511.0891 + 
                                     179.0427 * z.trans +
                                    1623.295  * abs.delta.T ) )
}


# Function for calculating drifting/blowing snow sublimation at a single height --------------

calc_bs_subl_local = function(qn, alpha, lambda, rh, Temp, rho, p, lsub, transient_term_local, Rv = 461.5){
  # Compute gamma function for size distribution
  gamma_alpha3      = gamma(alpha+3)
  gamma_alpha       = gamma(alpha)
  gamma_alpha_ratio = gamma_alpha3/gamma_alpha
  
  tempc = Temp - 273.15
  # saturation vapour pressure (Pa)
  tmp_real = calc_es_les(t = tempc)
  # Calculate dynamic viscosity of air (kg m-1 s-1)
  if(tempc >= 0){
    dyn_vis = (1.718+0.0049*tempc)*1e-5
  } else{
    dyn_vis = (1.718+0.0049*tempc-1.2e-5*tempc*tempc)*1e-5
  }
  
  # term in formula of best number (m-3)
  g = (1.333 * rho *  ( 918.0 - rho) * 9.81 ) / ( dyn_vis * dyn_vis )
  # Calculate mean best number (dimensionless)
  mean_NBest = g * gamma_alpha_ratio * (1.0/(lambda * lambda * lambda))
  # Calculate coefficients linking mean best number with Reynolds number 
  b_loc = b_re(mean_NBest)
  a_loc = a_re(mean_NBest,b_loc)
  
  # molecular diffusivity of water vapor
  diffu = 2.11e-5 * ((Temp/273.15)^(1.94)) * (101325/p)
  # thermal conductivity of air (W m-1 K-1)
  tcond = (5.69 + 0.0168*(tempc)) *1e-5 * 418.936
  
  # degree of undersaturation
  undsat = rh - 1.0
  # term used below (m W-1)
  omega = 1/(tcond*Temp) * (lsub/(Rv*Temp) - 1.0)
  # term in the Thorpe-Mason formula (kg m-1 s-1)
  A = 2.0*pi / ( (lsub*omega) + ( (Rv*Temp)/(diffu*tmp_real) ) )
  B = 0.2682 * A     # The factor of 0.2682 seems to be 0.308 * Schmidt_number**(1/3)
  # term in sublimation integrated over the number size distribution
  C = B * (a_loc**(0.5)) * (g ** (b_loc/2.0))
  # calculate sublimation of all blowing snow particles (kg kg-1 s-1) based on original TM formula
  tmp_SQ = qn * ( undsat * ( (0.78 * A * alpha / lambda) + C * (gamma(1.0+1.5*b_loc + alpha)/gamma_alpha) / (lambda**(1+1.5*b_loc)) ) )
  # change in heat storage integrated over range of particle diameters
  bs_storage_change = 0
  
  # Add a term for the change in heat storage of particles if required input is provided
  if(!is.null(transient_term_local)){
    f_storage = 1/6 * transient_term_local$rho_p * transient_term_local$c_p_p
    tmp_storage = f_storage * (alpha+2) * (alpha+1) * alpha * ( transient_term_local$intercept * lambda + transient_term_local$slope * (alpha+3) ) / lambda^4
    # additional term in sublimation formula
    SQ_term_2 = qn * A/2 * omega * tmp_storage
    tmp_SQ    = tmp_SQ + SQ_term_2
    # change in heat storage integrated over range of particle diameters (J kg-1 s-1)
    bs_storage_change = qn * pi * tmp_storage
  }
  
  # if(tmp_SQ <= 0) then{
  #   # If there is sublimation (tmp_SQ is negative), limit the absolute magnitude to the amount that causes
  #   # approximately saturation or the amount of blowing snow that is present
  #   bs_sublim_qi[k] = max(-qi_final[k]/dt_in, tmp_SQ, (bs_qv[k]-qsat)*0.999/dt_in)
  # } else{
  #   # Limit vapor deposition to the amount that causes near-saturation
  #   bs_sublim_qi[k] = MIN(tmp_SQ,DBLE((bs_qv[k]-qsat)*0.999/dt_in))
  # }
  
  return(data.frame(tmp_SQ, bs_storage_change))
}


# Function for calculating drifting/blowing snow sublimation ----------------------------------
# (and relative humidity, latent heat of sublimation, and layer thicknesses) 
# based on the original or modified Thorpe-Mason formula as (planned) in CRYOWRF

## Required functions:
# Lsubl()
# calc_RH_ice()
# a_re()
# b_re()

## Input:
# bs_Z          heights (m)
# Z_boundary    heights (m) of lower and upper boundaries between which drifting/blowing snow sublimation is calculated, vector of length 2
# sub_n_layers  number of saltation levels
# dz_salt       thickness (m) of grid cells in saltation layer
# bs_temp       air temperatures (K)
# bs_qv         specific humidities (kg kg-1)
# p             air pressure per height (Pa)
# RHO           air density per height (kg m-3)
# alpha         shape parameter of gamma distribution (dimensionless), vector of same length as bs_Z
# qi_final      blowing snow mass mixing ratios (kg kg-1)
# bs_meanR      mean radius of drifting/blowing snow particles (m) per height
# Rv            gas constant for water vapor (J kg-1 K-1)
# transient_term   a list of named variables if term for change in heat storage is to be parametrized (modified TM formula), otherwise NULL (original TM formula). The list contains:
#                  rho_p       density of the particle (kg m-3),
#                  c_p_p       specific heat capacity of the particle ( J kg-1 K-1),
#                  intercept   diameter-independent term in the linear parametrization of the change in particle temperature (dTdt), i.e. the term c in the formula dTdt = m d + c , where d is particle diameter.
#                              A vector of the same length as bs_Z. If both intercept and slope are zero, the result is identical to the original TM formula.
#                  slope       coefficient of the diameter-dependent-dependent term in the linear parametrization of dTdt, i.e. the term m in the formula above.
#                              A vector of the same length as bs_Z.
#                  gap         list with scalar elements named intercept and slope; for the height in the upper part of the cell around the saltation layer height

## Output:
# Data frame with sublimation (bs_sublim_qi, kg kg-1 s-1), 
#   change in heat storage integrated over range of particle diameters (bs_storage_change, J kg-1 s-1),
#   relative humidity (RH, %), latent heat of sublimation (lsub, J kg-1), depth of each layer (bs_grid_height, m), 
#   saturation specific humidity (bs_qsat, kg kg-1), blowing snow number mixing ratio (qni, kg-1)

calc_bs_subl = function(bs_Z, Z_boundary, sub_n_layers, dz_salt, bs_temp, bs_qv, p, RHO, alpha, qi_final, bs_meanR, h_salt, transient_term = NULL, q.gap = NA){
  # rate parameter of gamma distribution
  sub_lambda = alpha / (2*bs_meanR) # m-1
  # drifting snow number mixing ratio (kg-1)
  sub_qni    = qi_final * sub_lambda^3 * 6 / (pi*918.0) * gamma(alpha) / gamma(alpha+3)
  # number of grid levels
  bs_Ntot = length(bs_Z)
  # difference between heights
  # bs_dz = c(bs_Z[1] - Z_boundary[1], diff(bs_Z), Z_boundary[2] - bs_Z[length(bs_Z)])
  bs_dz = diff(bs_Z) #[1:sub_n_layers]), h_salt - bs_Z[sub_n_layers], bs_Z[sub_n_layers+1] - h_salt, diff(bs_Z[(sub_n_layers+1):length(bs_Z)]), Z_boundary[2] - bs_Z[length(bs_Z)])
  # depth per layer (m), including uppermost layer
  bs_grid_height = c(bs_Z[1] - Z_boundary[1] + 0.5*bs_dz[1],
                     0.5 * ( bs_dz[1:(bs_Ntot-2)] + bs_dz[2:(bs_Ntot-1)] ),
                     0.5*bs_dz[bs_Ntot-1] + Z_boundary[2] - bs_Z[bs_Ntot] )
  # bs_grid_height = c(bs_Z[1] - Z_boundary[1] + 0.5*bs_dz[1],                           # saltation mesh
  #                    0.5 * ( bs_dz[1:(sub_n_layers-2)] + bs_dz[2:(sub_n_layers-1)] ),  
  #                    bs_dz[sub_n_layers-1],                                            # for z = h_salt
  #                    bs_dz[sub_n_layers] - 0.5*bs_dz[sub_n_layers-1] + 0.5*bs_dz[sub_n_layers+1],                  # first level above h_salt
  #                    0.5 * ( bs_dz[(sub_n_layers+1):(bs_Ntot-2)] + bs_dz[(sub_n_layers+2):(bs_Ntot-1)] ), # fine mesh and coarse mesh
  #                    0.5*bs_dz[bs_Ntot-1] + Z_boundary[2] - bs_Z[bs_Ntot] )
  # latent heat of sublimation    
  lsub = Lsubl(bs_temp - 273.15)
  # relative humidity (1)
  bs_RH_loc   = 0.01 * calc_RH_ice(Temp = bs_temp, q = bs_qv, rho = RHO)
  # saturation specific humidity (kg kg-1)
  bs_qsat     = bs_qv / bs_RH_loc
  
  if(!is.null(transient_term)){
    if(length(transient_term$intercept) != length(bs_Z) | length(transient_term$slope) != length(bs_Z)) warning("length of transient_term$intercept or transient_term$slope is not length(bs_Z)")  
  }
  
  ## Compute sublimation
  bs_sublim_qi      = rep(0, length(qi_final))
  bs_storage_change = rep(0, length(qi_final))
  for(k in 1:bs_Ntot){
    # skip the height if temperature is NA
    if(is.na(bs_temp[k])) next
    # Calculate sublimation term (if blowing snow is present)
    if( qi_final[k] >= 1e-10 ){
      if(is.null(transient_term)){
        transient_term_local = NULL
      } else{
        transient_term_local           = transient_term
        transient_term_local$slope     = transient_term$slope[k]
        transient_term_local$intercept = transient_term$intercept[k]
      }
      # moisture exchange (kg kg-1 s-1) and particle heat storage change (J kg-1 s-1)
      my_exch = calc_bs_subl_local(qn = sub_qni[k], alpha = alpha[k], lambda = sub_lambda[k], rh = bs_RH_loc[k],
                                   Temp = bs_temp[k], rho = RHO[k], p = p[k], lsub = lsub[k], 
                                   transient_term_local = transient_term_local)
    } else{
      my_exch = data.frame(tmp_SQ = 0, bs_storage_change = 0)
    }
    
    # special treatment of layer just above uppermost saltation level
    if(k == sub_n_layers){
      # subdivide layer into 2 parts
      # thickness     = 0.5 * sum( bs_dz[(k-1):k] )
      thickness_gap = bs_grid_height[k] - dz_salt
      z_gap         = bs_Z[k] + 0.5*dz_salt + 0.5*thickness_gap
      # z_gap         = mean( bs_Z[(k-1):k] ) - 0.5*thickness_gap
      my.z =     bs_Z[k:(k+1)]
  
      if(is.na(q.gap)){
        # estimate blowing snow mixing ratios and size distribution for upper part ('gap') of the layer by interpolating
        my.q = qi_final[k:(k+1)]
        # assume exponential (for z_gap < h_salt) or power law function (for z_gap >= h_salt) for blowing snow mass mixing ratio
        if(z_gap < h_salt){
          log.q.gap = approx(x = my.z, y = log(my.q), xout = z_gap)$y
        } else{
          log.q.gap = approx(x = log(my.z), y = log(my.q), xout = log(z_gap))$y
        }
        # estimated mass mixing ratio (kg kg-1)
        q.gap = exp(log.q.gap)
      }
      
      if( q.gap >= 1e-10 ){
        # if(is.na(q.gap)){
          # same interpolation for number mixing ratio
          my.q = sub_qni[k:(k+1)]
          # assume exponential (for z_gap < h_salt) or power law function (for z_gap >= h_salt) for blowing snow number mixing ratio
          if(z_gap < h_salt){
            log.q.gap = approx(x = my.z, y = log(my.q), xout = z_gap)$y
          } else{
            log.q.gap = approx(x = log(my.z), y = log(my.q), xout = log(z_gap))$y
          }
          # estimated number mixing ratio (kg-1)
          qn.gap = exp(log.q.gap)
          # interpolate linearly the shape parameter of the size distribution
          alpha.gap  = approx(x = my.z, y = alpha[k:(k+1)], xout = z_gap)$y
          lambda.gap =  ( pi*918.0/6.0 * qn.gap/q.gap * gamma(alpha.gap+3) / gamma(alpha.gap) )**(1.0/3.0)
        
        # interpolate temperature assuming log profile
        temp_gap = approx(x = log(my.z), y = bs_temp[k:(k+1)], xout = log(z_gap))$y
        # interpolate spec humidity assuming log profile
        qv_gap   = approx(x = log(my.z), y = bs_qv[k:(k+1)], xout = log(z_gap))$y
        lsub_gap = Lsubl(temp_gap - 273.15) # (J kg-1 K-1)
        RH_gap   = 0.01 * calc_RH_ice(Temp = temp_gap, q = qv_gap, rho = RHO[k]) # (1)
        if(is.null(transient_term)){
          transient_term_local = NULL
        } else{
          transient_term_local$slope     = transient_term$gap$slope
          transient_term_local$intercept = transient_term$gap$intercept
        }
        # moisture exchange and particle heat storage change for the upper part of the layer
        my_exch_gap = calc_bs_subl_local(qn = qn.gap, alpha = alpha.gap, lambda = lambda.gap, rh = RH_gap,
                                         Temp = temp_gap, rho = RHO[k], p = p[k], lsub = lsub_gap, 
                                         transient_term_local = transient_term_local)
        # Weighted average of the two parts of the layer
        my_exch$tmp_SQ            = ( my_exch$tmp_SQ            * dz_salt + my_exch_gap$tmp_SQ            * thickness_gap ) / bs_grid_height[k]
        my_exch$bs_storage_change = ( my_exch$bs_storage_change * dz_salt + my_exch_gap$bs_storage_change * thickness_gap ) / bs_grid_height[k]
      }
    }
    
    if(abs(my_exch$tmp_SQ) <= 1.0e-8){
      my_exch$tmp_SQ = 0.0
    }
    # store results
    bs_sublim_qi[k]      = my_exch$tmp_SQ
    bs_storage_change[k] = my_exch$bs_storage_change

  }
  out = data.frame(bs_sublim_qi, bs_storage_change, RH = bs_RH_loc*100, lsub, bs_grid_height, bs_qsat, qni = sub_qni)
  return(out)
}
