# Function for explicit solver of 1D diffusive transport on staggered grid --------------------------

# x        quantity of interest at half nodes
# dz_half  differences between half nodes (m), either a single number or a vector of length 'length(x) - 1'
# dz_full  differences between full nodes (m), either a single number or a vector of the same length as x
# K        diffusivity (m2 s-1) at full nodes, length of this vector is length(x)+1
# BC1,BC2  Boundary condition for first (lower) and second (upper) boundary of domain, respectively. List of length 2: first element is the numeric value of the condition, second element is character specifying the type (either Neumann "NEU" or flux "FLX")

# output: list with 3 elements: change of quantity of interest per second,
                              # flux at lower boundary,
                              # flux at upper boundary.

solve_diffusion = function(x, dz_half, dz_full = dz_half, K, BC1, BC2){
  # Diffusive flux at full nodes
  flux = -K[2:(length(K)-1)] * diff(x) / dz_half
  # BC1
  flux1 = BC1[[1]] # assuming that it specifies the flux (FLX)
  if(BC1[[2]] == "NEU"){
    flux1 = -K[1] * flux1
  } else{
    if(BC1[[2]] != "FLX"){
      stop("type of boundary condition is neither NEU nor FLX")
    }
  }
  # BC2
  flux2 = BC2[[1]] # assuming that it specifies the flux (FLX)
  if(BC2[[2]] == "NEU"){
    flux2 = -K[length(K)] * flux2
  } else{
    if(BC2[[2]] != "FLX"){
      stop("type of boundary condition is neither NEU nor FLX")
    }
  }
  flux_all = c(flux1, flux, flux2)
  # change of quantity of interest per second
  change = -diff(flux_all) / dz_full
  return(list(change = change, flux_BC1 = flux1, flux_BC2 = flux2))
}


# Function for computing a quantity at next higher half node above domain ------------------------------------------
# using upper boundary condition

# x_UBC           upper boundary condition, list of 2 elements (value and character, "DRC" or "NEU")
# x_just_below    quantity of interest at uppermost half node within domain
# dz              distance between the uppermost half node within domain and the top boundary

apply_UBC = function(x_UBC, x_just_below, dz = NA){
  if(x_UBC[[2]] == "NEU"){
    x_last = x_just_below + x_UBC[[1]] * 2*dz
  } else{
    if(x_UBC[[2]] == "DRC"){  # boundary condition refers to full node corresponding to upper boundary
      x_last = x_just_below + (x_UBC[[1]] - x_just_below) * 2
    } else{
      stop("type of boundary condition is neither NEU nor DRC")
    }
  }
}


# Function for explicit solver for vertical blowing snow diffusion and sedimentation (similar to CRYOWRF 1D) --------

# q_half        blowing snow mass mixing ratio at half nodes (kg kg-1
# qn_half       blowing snow number mixing ratio at half nodes (kg-1)
# alpha         shape parameter of gamma distribution (dimensionless) at half nodes, vector of same length as q_half
# temp          air temperature (K) at all half nodes in the domain
# z_half        height (m) of all half nodes in the domain
# in_susp       indexes of suspension levels (for half nodes)
# z_full_susp   height (m) of full nodes in suspension layer
# K             diffusivity (m2 s-1) at full nodes in suspension layer, length of this vector is length(in_susp)+1
# q_top, qn_top   Upper Boundary condition for blowing snow mass or number mixing ratio. List of length 2: first element is the numeric value of the condition, second element is character specifying the type (Neumann "NEU", Dirichlet "DRC")
# temp_top        Upper Boundary condition for air temperature. List of length 2: first element is the numeric value of the condition, second element is character specifying the type (Neumann "NEU", Dirichlet "DRC")
# alpha_top       Shape parameter of gamma distribution of particle size at upper boundary. Single numeric value.
# rho             Air density (kg m-3), single value, assumed to be constant here
# sedim_diff      Character specifying the type of finite difference used for the sedimentation term, current options are "forward", "central", "q_forward_v_backward", i.e., considering respectively the difference between the half level of interest and the next higher half level; 
#                 interpolated mixing ratios and temperature at full levels; mixing ratios at the half level of interest and the next higher half level and fall velocities at next lower half level and half level of interest

# output: data frame with change of blowing snow mass and number mixing ratios per second (q_change and qn_change, respectively)

bs_diffu_sedim_1D = function(q_half, qn_half, alpha, temp, z_half, in_susp, z_full_susp, K, q_top, qn_top, temp_top, alpha_top, rho, sedim_diff){
  
  # height difference between half nodes in suspension layer
  dz_half = diff(z_half[in_susp])
  # Height difference between full nodes in suspension layer (cell thickness)
  dz_full = diff(z_full_susp) 
  # Height difference between saltation layer height (half node) and next higher half node
  dz1 = diff(z_half)[in_susp[1]-1]
  
  # mixing ratios in suspension layer
  q  =  q_half[in_susp]
  qn = qn_half[in_susp]
  # blowing snow mass or number mixing ratio at saltation layer height
  q_salt  =  q_half[in_susp[1]-1]
  qn_salt = qn_half[in_susp[1]-1]
  
  # lower boundary condition: diffusive flux between saltation layer height (half node) and next higher half node 
  q_flux1 = -K[1] * (q[1] - q_salt) / dz1 # kg kg-1 m s-1
  q_BC1   = list(q_flux1, "FLX")
  
  qn_flux1 = -K[1] * (qn[1] - qn_salt) / dz1 # (kg-1 m s-1)
  qn_BC1 = list(qn_flux1, "FLX")
  
  # upper boundary condition: diffusive flux between half nodes just below and above the domain boundary
  if(q_top[[2]] == "DRC"){ # boundary condition refers to half node just above to upper boundary
    q_flux2 = -K[length(K)] * (q_top[[1]] - q[length(q)]) / tail(dz_full, n=1)
    q_BC2   = list(q_flux2, "FLX")
  } else{
    q_BC2 = q_top
  }
  if(qn_top[[2]] == "DRC"){ # boundary condition refers to half node just above to upper boundary
    qn_flux2 = -K[length(K)] * (qn_top[[1]] - qn[length(qn)]) / tail(dz_full, n=1)
    qn_BC2   = list(qn_flux2, "FLX")
  } else{
    qn_BC2 = qn_top
  }
  
  # Diffusion effect: change of mixing ratio per second
  q_diffu  = solve_diffusion(q,  dz_half, dz_full, K, BC1 = q_BC1,  BC2 = q_BC2)
  qn_diffu = solve_diffusion(qn, dz_half, dz_full, K, BC1 = qn_BC1, BC2 = qn_BC2)
  
  # q at next higher half node above domain
  if(q_top[[2]] == "NEU"){
    q_last    = apply_UBC(x_UBC = q_top,  x_just_below = q_half[length(q_half)],         dz = 0.5*dz_full[length(dz_full)])
  } else{ # DRC
    q_last    = q_top[[1]]
  }

  if(qn_top[[2]] == "NEU"){
    qn_last   = apply_UBC(x_UBC = qn_top, x_just_below = qn_half[length(qn_half)],       dz = 0.5*dz_full[length(dz_full)])
  } else{ # DRC
    qn_last   = qn_top[[1]]
  }

  # warn if there are negative mixing ratios at half node above domain
  if(q_last < 0 | qn_last < 0) warning("Negative mixing ratio of blowing snow at half node just above domain.")
  
  # temperature at next higher half node above domain
  temp_last = apply_UBC(x_UBC = temp_top, x_just_below = temp[length(temp)], dz = 0.5*dz_full[length(dz_full)])

  my_q  = c(q_half,  q_last)
  my_qn = c(qn_half, qn_last)
  my_z  = c(z_half, tail(z_half, n=1) + dz_full[length(dz_full)])
  my_temp = c(temp, temp_last)
  my_alpha = c(alpha, alpha_top)
  
  if(sedim_diff == "central"){
    # Avoid mixing ratios of zero such that we can apply the logarithm for interpolation purposes
    my_q[my_q <= 0]   = 1e-12
    my_qn[my_qn <= 0] = 1
    # interpolate from half nodes to full nodes assuming power law function for blowing snow mixing ratio
    log_q_full_power = approx(x = log(my_z), y = log(my_q),  xout = log(z_full_susp))$y
    log_qn_full_power= approx(x = log(my_z), y = log(my_qn), xout = log(z_full_susp))$y
    q_sedim    = exp( log_q_full_power )
    qn_sedim   = exp( log_qn_full_power )
    # interpolate temperature from half nodes to full nodes assuming log function
    temp_sedim = approx(x = log(my_z), y = my_temp, xout = log(z_full_susp))$y
    # shape parameter of gamma distribution of particle size, interpolate linearly
    alpha_sedim = approx(x = my_z, y = my_alpha, xout = z_full_susp)$y
    # terminal fall velocity (at full nodes)
    V_terminal = calc_vel_sedim(temp = temp_sedim, qi = q_sedim, qni = qn_sedim, alpha = alpha_sedim, RHO = rho)
  } else{
    if(sedim_diff == "forward"){
      q_sedim    = my_q[  in_susp[1]:length(my_q) ]
      qn_sedim   = my_qn[ in_susp[1]:length(my_qn) ]
      temp_sedim = my_temp[ in_susp[1]:length(my_temp) ]
      alpha_sedim= my_alpha[ in_susp[1]:length(my_alpha) ]
      # terminal fall velocity (at half nodes)
      V_terminal = calc_vel_sedim(temp = temp_sedim, qi = q_sedim, qni = qn_sedim, alpha = alpha_sedim, RHO = rho)
    } else{
      if(sedim_diff == "q_forward_v_backward"){
        # mixing ratios at half node of interest and next higher half node
        q_sedim    = my_q[  in_susp[1]:length(my_q) ]
        qn_sedim   = my_qn[ in_susp[1]:length(my_qn) ]
        in_vel     = (in_susp[1]-1) : (length(my_z)-1)
        # terminal fall velocity (at next lower half node and half node of interest)
        V_terminal = calc_vel_sedim(temp = my_temp[in_vel], qi = my_q[in_vel], qni = my_qn[in_vel], alpha = my_alpha[in_vel], RHO = rho)
      } else{
        if(sedim_diff == "q_backward_v_forward"){
          # mixing ratios at half node of interest and next higher half node
          q_sedim    = my_q[  (in_susp[1]-1) : (length(my_z)-1) ]
          qn_sedim   = my_qn[ (in_susp[1]-1) : (length(my_z)-1) ]
          in_vel     = in_susp[1]:length(my_z)
          # terminal fall velocity (at next lower half node and half node of interest)
          V_terminal = calc_vel_sedim(temp = my_temp[in_vel], qi = my_q[in_vel], qni = my_qn[in_vel], alpha = my_alpha[in_vel], RHO = rho)
        } else{
          warning("sedim_diff is unknown.")
        }
      }
    }
  }
  # VQ, VN   (mass and number weighted, resp.) terminal fall velocity (m s-1)
  VQ = V_terminal$VQ
  VN = V_terminal$VN
  
  in.curr  = 1:length(q)
  in.above = 1 + in.curr
  
  # Sedimentation effect: change of mixing ratio per second
  q_change_sedim  = ( q_sedim[in.above]  * VQ[in.above] - q_sedim[in.curr]  * VQ[in.curr] ) / dz_full
  qn_change_sedim = ( qn_sedim[in.above] * VN[in.above] - qn_sedim[in.curr] * VN[in.curr] ) / dz_full
  # Note: Division by distance between half nodes would lead to problem of unclosed mass balance in suspension domain because the sum of all considered dz is 10.75 m instead of '9 m - (0.2525 m + 0.5*h_salt)')
  
  output =       list("q_change" = q_diffu$change + q_change_sedim,
                      "qn_change"= qn_diffu$change + qn_change_sedim,
                      "q_flx_diffu_BC1" = q_diffu$flux_BC1,                               # kg kg-1 m s-1 # negative means downward flux
                      "q_flx_diffu_BC2" = q_diffu$flux_BC2,                               # kg kg-1 m s-1 # negative means downward flux
                      "q_flx_sedim_BC1" = - q_sedim[1] * VQ[1],                           # kg kg-1 m s-1 # negative means downward flux
                      "q_flx_sedim_BC2" = - q_sedim[length(q_sedim)] * VQ[length(q_sedim)], # kg kg-1 m s-1 # negative means downward flux
                      "qn_flx_diffu_BC1" = qn_diffu$flux_BC1,                                 # kg-1 m s-1 # negative means downward flux
                      "qn_flx_diffu_BC2" = qn_diffu$flux_BC2,                                 # kg-1 m s-1 # negative means downward flux
                      "qn_flx_sedim_BC1" = - qn_sedim[1] * VN[1],                             # kg-1 m s-1 # negative means downward flux
                      "qn_flx_sedim_BC2" = - qn_sedim[length(qn_sedim)] * VN[length(qn_sedim)]) # kg-1 m s-1 # negative means downward flux
return(output)
}


# Function for calculating mass and number weighted terminal fall velocities ----------------------
# To calculate V argument in bs_diffu_sedim_1D function
# using a_re and b_re functions from func_calc_bs_subl_TM.R

# temp     air temperature (K)
# qi       blowing snow mass mixing ratio (kg kg-1)
# qni      blowing snow number mixing ratio (kg-1)
# alpha    shape parameter of gamma distribution, either of length 1 or same length as qi
# RHO      air density vector (single value, assumed to be constant here)

# Output   data frame with column names VQ and VN for mass-weighted and number-weighted terminal fall velocity, respectively.

calc_vel_sedim = function(temp, qi, qni, alpha = 3, RHO = 1){
  if(length(alpha) == 1) alpha = rep(alpha, length(qi))
  # Compute gamma function for size distribution
  gamma_alpha3      = gamma(alpha+3)
  gamma_alpha       = gamma(alpha)
  gamma_alpha_ratio = gamma_alpha3/gamma_alpha
  # preallocate
  VQ = rep(NA, length(qi))
  VN = rep(NA, length(qi))
  # loop over height levels
  for(k in 1:length(qi)){
    # skip the height if temperature is NA
    #if(is.na(temp[k])) next
    tempc = temp[k] - 273.15
    # if mixing ratio
    if(tempc >= 0){
      dyn_vis = (1.718+0.0049*tempc)*1e-5
    } else{
      dyn_vis = (1.718+0.0049*tempc-1.2e-5*tempc*tempc)*1e-5
    }
    # if enough blowing snow is present, calculate terminal fall velocities, otherwise use low value of 0.001
    if( qi[k] > 1e-10 & qni[k] > 1e-5 ){
      lambda = ( pi*918.0/6.0 * qni[k]/qi[k] * gamma_alpha_ratio[k] )**(1.0/3.0)
      # term in formula of best number (m-3)
      g = (1.333 * RHO *  ( 918.0 - RHO) * 9.81 ) / ( dyn_vis * dyn_vis )
      # Calculate mean best number (dimensionless)
      mean_NBest = g * ( gamma_alpha_ratio[k] ) * (1.0/(lambda * lambda * lambda))
      # Calculate coefficients linking mean best number with Reynolds number 
      b_loc = b_re(mean_NBest)
      a_loc = a_re(mean_NBest,b_loc)
      # term used below (correcting bug in CRYOWRF, i.e. replacing 1.0 by air density)
      eta = a_loc * dyn_vis * g**b_loc / RHO
      VN[k] = eta * gamma(3.0*b_loc-1.0+alpha[k])/gamma_alpha[k]  * lambda**(1.0-3.0*b_loc)
      VQ[k] = eta * gamma(3.0*b_loc+2.0+alpha[k])/gamma_alpha3[k] * lambda**(1.0-3.0*b_loc)
    } else{
      # number weighted terminal fall velocity
      VN[k] = 0.001
      # mass weighted terminal fall velocity
      VQ[k] = 0.001
    }
  }
  return(data.frame("VQ" = VQ, "VN" = VN))
}




