# Script for running the one-dimensional model
# Model settings can be changed in config_parametrize.R

# Armin Sigmund

# Settings ----------------------------------------------------------------
rm(list=ls())

time.start = Sys.time()
# Choose one of the following four simulation cases
case = "case1_dx10cm"
# case = "case2_dx10cm"
# case = "case3a_dx10cm"
# case = "case3b_dx10cm"

# working directory
setwd("~/Documents/PhD_work/my_paper_drafts/paper_parametrization/data_for_EnviDat/1D_model_code/")

# path to functions
path.func = "func/"
source(file = file.path(path.func,"func_Lsubl.R"))
source(file = file.path(path.func,"func_MO_bulk_fluxes.R"))
source(file = file.path(path.func,"func_calc_es_les_RH_ice.R"))
source(file = file.path(path.func,"func_calc_q_bs_in_saltation.R"))
source(file = file.path(path.func,"func_calc_bs_subl.R"))
source(file = file.path(path.func,"func_solve_diffusion.R"))
source(file = file.path(path.func,"func_calc_h_salt.R"))
source(file = file.path(path.func,"func_calc_ustar_raupach.R"))
source(file = file.path(path.func,"func_solve_poly3.R"))
# get constants like gas constants and specific heat capacity of air
source(file = "config_constants.R")
# get model settings
source(file = "config_1D_model.R")
# Initialize the warning flag
flag.warn = list(sublimation = T, deposition = T) # T means that we get a warning if RH becomes > or < 100 % due to sublimation or deposition, respectively, for the first time, otherwise no warning
  
# Simulate sublimation with time ... -------------------------------------

# set variable to NA, which is only used in another model set-up
if(impose.ustar){
  vw_coarse_first = NA
} else{
  ustar.fix = NA
}

# load LES data on LES mesh (for estimating blowing snow boundary conditions [if blowing snow is parametrized] and later for plotting LE and H profiles)
load(file = file.forcings.LES.mesh)
conc = run1$ptcl$mass.conc
d.avg = data.frame(z = run1$ta1$z, d = run1$ptcl$d_p_mean)

# Upper boundary conditions for mass and number mixing ratios of blowing snow (at half node just above the domain of the 1D model; needed if we parametrize blowing snow mixing ratio and for plotting)
# fit power law profile to heights from 2 to 5 m and extrapolate
z.range = c(2,5)
in.z.range = which(run1$ptcl$z < z.range[2] & run1$ptcl$z > z.range[1])
my.var = run1$ptcl$mass.conc[in.z.range] / rho_sub
my.x = log(run1$ptcl$z[in.z.range])
my.mod.mass = lm( log(my.var) ~ my.x )
var.pred = predict( my.mod.mass, newdata = list(my.x = log(z_top_boundary+0.5*z_firstlev)) )
q.bs.top = exp(var.pred)    #approx(x = run1$ptcl$z, y = run1$ptcl$mass.conc / rho_sub, xout = z_top_boundary+0.5*z_firstlev)$y
qi_top   = list( q.bs.top, "DRC" ) # (kg kg-1)
my.var = run1$ptcl$number.conc[in.z.range] / rho_sub
my.mod = lm( log(my.var) ~ my.x )
var.pred = predict( my.mod, newdata = list(my.x = log(z_top_boundary+0.5*z_firstlev)) )
qn.bs.top = exp(var.pred)   #approx(x = run1$ptcl$z, y = run1$ptcl$number.conc / rho_sub, xout = z_top_boundary+0.5*z_firstlev)$y
qni_top  = list( qn.bs.top , "DRC" )
# zero mixing ratios at top
# qi_top   = list( 0, "DRC" ) # (kg kg-1)
# qni_top  = list( 0, "DRC" )

# shape parameter of gamma distribution of particle size, computed from LES data
alpha_LES = run1$ptcl$gamma.shape

# shift upper boundary conditions closer to lower boundary conditions (for sensitivity test)
if(f.shift.BC.top != 0){ print("Shifting upper T and q boundary conditions") }
BC_q_top[[1]] = BC_q_top[[1]] - f.shift.BC.top * (BC_q_top[[1]] - QSFC)
BC_T_top[[1]] = BC_T_top[[1]] - f.shift.BC.top * (BC_T_top[[1]] - TSK)

if(!is.numeric(Z_fine)){ # create 40 fine levels with logarithmic spacing
  if(!is.numeric(parametrize.h_salt)) stop("Both Z_fine and parametrize.h_salt are not numeric.")
  z.shift = 0.25
  my.transform = function(x, offset){ log(offset + x) }
  dz_bot_log = my.transform(dz_salt, z.shift) - my.transform(0, z.shift)
  Z_fine = exp( seq( my.transform(0, z.shift) + dz_bot_log, 0.95*my.transform(z_firstlev/2 - parametrize.h_salt, z.shift), dz_bot_log ) ) - z.shift + parametrize.h_salt
}
# estimate fluxes via MOST (ustar is needed to determine h_salt, LE and H may be needed for initial T and q profiles)
if(corr.stability){ # with stability correction
  flx = calc_fluxes_iter(z_ref_vw = 0.5*z_firstlev, rough_len_m = rough_len_m, rough_len_t = rough_len_t, rough_len_q = rough_len_q, vw_ref = vw_coarse_first,
                         T_ref = temp_coarse_init[1], qv_ref = qv_coarse_init[1], T_surf = TSK, qv_surf = QSFC, prescribe_ustar = ustar.fix)
} else{ # assume neutral
  # surface fluxes
  flx         = data.frame(u.star  = if(impose.ustar){ ustar.fix }else{ vk * vw_coarse_first / ( log(0.5*z_firstlev / rough_len_m) ) },
                           zeta    = 0,
                           psi.m   = 0,
                           psi.s   = 0)
  flx$Tw_flux = -vk * flx$u.star * (temp_coarse_init[1] - TSK) / log( 0.5*z_firstlev / rough_len_t )
  flx$qw_flux = -vk * flx$u.star * (qv_coarse_init[1] - QSFC)  / log( 0.5*z_firstlev / rough_len_q )
}
# height of saltation layer (m) 
if(is.numeric(parametrize.h_salt)){
  h_salt = parametrize.h_salt # assume fixed height, e.g. 0.1275 m
  # only use fine levels above h_salt
  Z_fine = Z_fine[Z_fine > h_salt]
} else{ # Pomeroy and Male (1992) or Lehning et al., 2008
  h_salt = calc_h_salt(method = parametrize.h_salt, ustar = flx$u.star, z0 = rough_len_m)
}
# choose the value among the following ones, which is closest to h_salt and lower than the first fine level: x * 1.5 cm + 0.75 cm with x = 0, 1, 2, ...
h_candidates = seq(0.5*dz_salt, Z_fine[1]-0.5*dz_salt, dz_salt)
h_salt = h_candidates[ which.min(abs(h_candidates - h_salt)) ]

Z_top = z_firstlev * 0.5 # corresponds to the first WRF U-level?!
if(use_salt_levels){
  # heights of saltation mesh
  if(only1salt.level){
    salt_Z = 0.5*dz_salt #d.avg$z[1]
  } else{
    salt_Z = seq(0.5*dz_salt, h_salt+0.5*dz_salt, dz_salt) #d.avg$z[which(d.avg$z < (h_salt+0.5*dz_salt))]
  }
} else{
  salt_Z = NULL
}
# heights of saltation mesh and fine mesh
snc_Z = c( salt_Z, Z_fine )
# number of layers in fine mesh and saltation mesh
snc_N = length(snc_Z)
dz_coarse = z_firstlev
# heights of coarse mesh
Z_coarse = seq(Z_top, z_top_boundary, dz_coarse)
# all heights
bs_Z = c(snc_Z, Z_coarse)
bs_Ntot = length(bs_Z)
# number of layers in saltation
sub_n_layers = length(salt_Z) #sum(bs_Z < (h_salt+0.5*dz_salt))
# heights (m) of half nodes in suspension layer
z_susp = bs_Z[(sub_n_layers+1):length(bs_Z)]
# air density
RHO = rep(rho_sub, bs_Ntot)
# height differences
bs_dz = diff(bs_Z)
# heights of full nodes
if(T.q.option == "prognostic"){
  z_full_K = c(0, bs_Z[1:length(bs_dz)] + 0.5*bs_dz, z_top_boundary)
} else{
  z_full_K = seq(Z_top - dz_coarse/2, by = dz_coarse, length.out =length(Z_coarse)+1)
}

if(parametrize.qi == "all"){
  if(!parametrize.int.salt.conc){
    in.z.fit = which(run1$ta1$z > 0.01 & run1$ta1$z < 0.15)
    # fit exponential profile to mass mixing ratio in LES data up to h_salt
    my_fit = lm( log(run1$ptcl$mass.conc[in.z.fit] / rho_sub) ~ run1$ta1$z[in.z.fit] )
    # integrated saltation mass concentration (kg m-2) # integral of the exponential profile from z=0 to z=infinity
    drift_conc_LES = -rho_sub / coef(my_fit)[2] * exp( coef(my_fit)[1] )
  }
}

# Define friction velocity at each full node (and surface friction velocity if applicable)
if(ustar.Raupach == "no"){ # constant with height
  my.ustar = rep(flx$u.star, length(z_full_K))
  ustar.s  = ifelse(impose.ustar, ustar.fix, NA) # needed for calculation of surface heat and moisture fluxes
} else{ # compute friction velocity profile according to Raupach (1991)
  if(ustar.Raupach == "LES-based"){
    # surface friction velocity
    ustar.s = run1$boundary.flux$ustar_surf
  } else{
    if(parametrize.int.salt.conc){
      # surface friction velocity = threshold friction velocity (m s-1)
      ustar.s = ustar_thresh = run1$boundary.flux$ustar_surf
    } else{
      a_coef = 2.6; b_coef = 2.5; c_coef = 2
      my_term = 2.8 * drift_conc_LES * 9.81 / (rho_sub * flx$u.star^2)
      my_ustar_ratio = solve_poly3(c(-b_coef, -c_coef, b_coef-a_coef, c_coef-my_term, a_coef), c(1, flx$u.star/0.1))
      # surface friction velocity = threshold friction velocity (m s-1)
      ustar.s = ustar_thresh = flx$u.star / my_ustar_ratio
    }
  }
  # friction velocity (m s-1) at full nodes
  my.ustar = calc_ustar_raupach(z = z_full_K, ustar = flx$u.star, ustar.t = ustar.s, h.salt = h_salt)
}

# mean particle radius (m) per height in suspension layer (from LES)
meanR_susp = approx(x = run1$ptcl$z, y = 0.5*run1$ptcl$d_p_mean, xout = z_susp)$y
if(use_salt_levels){
  if(parametrize.d){ # assume mean particle radius (m) in saltation layer
    dp.salt.layer = approx(c(bs_Z[1], h_salt), c(dp.salt.first, dp.hsalt), xout = salt_Z, rule = 2)$y
  } else{ # if particle radius in saltation layer is taken  from LES results
    dp.salt.layer = d.avg$d[1:sub_n_layers]
  }
} else{
  dp.salt.layer = NULL
}
# (initial) mean particle radius per height
meanR = c( 0.5 * dp.salt.layer, meanR_susp )

# shape parameter of gamma distribution of particle size at half nodes
if(parametrize.alpha){
  alpha = rep(alpha.const, bs_Ntot)
  alpha_ub = alpha.const
} else{
  alpha = approx(x = d.avg$z, y = alpha_LES, xout = bs_Z, rule = 2)$y
  alpha_ub = approx(x = d.avg$z, y = alpha_LES, xout = z_top_boundary+0.5*z_firstlev )$y
}
# initialize the drifting snow mass mixing ratio (kg kg-1)
if(parametrize.qi == "all"){
  qi_final = rep(0, bs_Ntot)
  qni_final= rep(0, bs_Ntot)
} else{ 
  if(parametrize.qi == "suspension"){
    qi_salt = conc[1:sub_n_layers] / rho_sub # kg kg-1
    qi_final = c( qi_salt, rep(0, bs_Ntot - sub_n_layers) )
    sub_alpha  = alpha[1:sub_n_layers]
    # rate parameter of gamma distribution
    sub_lambda = sub_alpha / (2*meanR[1:sub_n_layers]) # m-1
    # estimate number concentration (kg-1) based on mean particle radius and assumed shape parameter of gamma distribution 
    qni_salt = qi_final[1:sub_n_layers] * sub_lambda^3 * 6 / (pi*918.0) * gamma(sub_alpha)/gamma(sub_alpha+3)
    qni_final= c( qni_salt, rep(0, bs_Ntot - sub_n_layers) ) # kg-1
  } else{ # prescribe drifting snow mass mixing ratio (kg kg-1)
    # mass mixing ratio from LES (linearly interpolated below height of 5 m, extrapolated above using power law)
    qi_susp = approx(x = run1$ptcl$z, y = run1$ptcl$mass.conc / rho_sub, xout = z_susp[z_susp <= 5])$y
    var.pred = predict( my.mod.mass, newdata = list(my.x = log(z_susp[z_susp > 5])) )
    qi_susp = c(qi_susp, exp(var.pred))
    if(use_salt_levels){
      qi_final = c( conc[1:sub_n_layers] / rho_sub, qi_susp) # (kg kg-1)
    } else{
      qi_final = qi_susp
    }
  }
}

## Initial conditions
# temperature and specific humidity profiles in fine and saltation meshes
if(T.q.option == "prognostic"){
  temp_snc = calc_MO_profile(z_ref = rough_len_t, x_ref = TSK,  x_star = -flx$Tw_flux / flx$u.star, psi = flx$psi.s, z_out = snc_Z)
  qv_snc   = calc_MO_profile(z_ref = rough_len_q, x_ref = QSFC, x_star = -flx$qw_flux / flx$u.star, psi = flx$psi.s, z_out = snc_Z)
} else{
  temp_snc = rep(NA, snc_N)
  qv_snc   = rep(NA, snc_N)
}
bs_temp     = c( temp_snc, temp_coarse_init )
bs_qv     = c( qv_snc, qv_coarse_init )
# assume constant pressure (Pa) based on ideal gas law and surface conditions
press = RHO * R.dry.air * (1 + 0.608*QSFC) * TSK

# allocate arrays/data frames for storing results
n.rows = ceiling(nstep_max*dt_in/interval.out)
out = array(data = NA, dim = c(n.rows, bs_Ntot, 6))
if(parametrize.qi != "no"){
  nrows.qi = n.rows
  ncol.bound = 14
} else{
  nrows.qi = 1
  ncol.bound = 4
}
out.qi = out.qni = matrix(nrow = nrows.qi, ncol = bs_Ntot)
out.bound = as.data.frame(matrix(nrow = n.rows, ncol = ncol.bound))
# initialize time (s)
t = 0
## time loop
for(i in 1:nstep_max){
  t = t + dt_in
  ## update friction velocity if necessary
  if(!impose.ustar & corr.stability){
    # estimate friction velocity from wind speed etc. at first coarse level
    flx = calc_fluxes_iter(z_ref_vw = bs_Z[snc_N+1], rough_len_m = rough_len_m, rough_len_t = rough_len_t, rough_len_q = rough_len_q, vw_ref = vw_coarse_first,
                           T_ref = bs_temp[snc_N+1], qv_ref = bs_qv[snc_N+1], T_surf = TSK, qv_surf = QSFC)
    if(ustar.Raupach != "no"){ # compute friction velocity profile according to Raupach (1991)
      # friction velocity (m s-1) at full nodes
      my.ustar = calc_ustar_raupach(z = z_full_K, ustar = flx$u.star, ustar.t = ustar.s, h.salt = h_salt)
    } else{ # constant with height
      my.ustar = rep(flx$u.star, length(z_full_K))
    }
  }
  # eddy diffusivity (m s-1), defined at full levels
  K_eddy = vk * my.ustar * z_full_K
  ## determine surface moisture and heat fluxes via MO bulk formula considering surface friction velocity
  index_1 = ifelse(T.q.option == "prognostic", 1, snc_N+1) # use T and q at first saltation level or first coarse level
  if(corr.stability){
    surf_flx = calc_fluxes_iter(z_ref_vw = bs_Z[snc_N+1], z_ref_scalar = bs_Z[index_1], rough_len_m = rough_len_m, rough_len_t = rough_len_t, rough_len_q = rough_len_q,
                                vw_ref = vw_coarse_first, T_ref = bs_temp[index_1], qv_ref = bs_qv[index_1], T_surf = TSK, qv_surf = QSFC, prescribe_ustar = ustar.s)
  } else{ # assume neutral
    surf_flx = data.frame(Tw_flux = vk * ustar.s / log(bs_Z[index_1]/rough_len_t) * ( TSK - bs_temp[index_1] ),
                          qw_flux = vk * ustar.s / log(bs_Z[index_1]/rough_len_q) * ( QSFC - bs_qv[index_1] ))
  }
  # T and q surface fluxes are not sensitive with respect to accounting for or neglecting stability correction between surface and first saltation level (like in latest LES version)
  BC_T_bottom = list(surf_flx$Tw_flux, "FLX")
  BC_q_bottom = list(surf_flx$qw_flux, "FLX")
  
  ## construct T and q profiles in fine mesh and saltation mesh (different options)
  if(T.q.option == "linear_without_salt_layer"){
    bs_temp[(sub_n_layers+1):snc_N] = approx(x = c(0, bs_Z[snc_N+1]), y = c(TSK, bs_temp[snc_N+1]), xout = snc_Z[(sub_n_layers+1):snc_N])$y
    bs_qv[(sub_n_layers+1):snc_N]   = approx(x = c(0, bs_Z[snc_N+1]), y = c(QSFC, bs_qv[snc_N+1]),  xout = snc_Z[(sub_n_layers+1):snc_N])$y
  } else{
    if(T.q.option == "linear"){
      bs_temp[1:snc_N]              = approx(x = c(0, bs_Z[snc_N+1]), y = c(TSK, bs_temp[snc_N+1]), xout = snc_Z)$y
      bs_qv[1:snc_N]                = approx(x = c(0, bs_Z[snc_N+1]), y = c(QSFC, bs_qv[snc_N+1]),  xout = snc_Z)$y
    } else{
      if(T.q.option == "MO-bulk"){
        # compute current temperature and humidity scales using first coarse level
        if(corr.stability){ # with stability correction
          flx = calc_fluxes_iter(z_ref_vw = bs_Z[snc_N+1], rough_len_m = rough_len_m, rough_len_t = rough_len_t, rough_len_q = rough_len_q, vw_ref = vw_coarse_first,
                                 T_ref = bs_temp[snc_N+1], qv_ref = bs_qv[snc_N+1], T_surf = TSK, qv_surf = QSFC, prescribe_ustar = ustar.fix)
        } else{ # assume neutral
          flx$Tw_flux = -vk * flx$u.star * (bs_temp[snc_N+1] - TSK) / log( bs_Z[snc_N+1] / rough_len_t )
          flx$qw_flux = -vk * flx$u.star * (bs_qv[snc_N+1] - QSFC)  / log( bs_Z[snc_N+1] / rough_len_q )
        }
        # temperature (K) based on MOST
        bs_temp[1:snc_N] = calc_MO_profile(z_ref = rough_len_t, x_ref = TSK, x_star = -flx$Tw_flux / flx$u.star, psi = flx$psi.s, z_out = snc_Z)
        # specific humidity (kg kg-1)
        bs_qv[1:snc_N]   = calc_MO_profile(z_ref = rough_len_q, x_ref = QSFC, x_star = -flx$qw_flux / flx$u.star, psi = flx$psi.s, z_out = snc_Z)
      } else{
        if(T.q.option != "prognostic"){
          warning("T.q.option unknown")
        }
      }
    }
  }
  # parametrize saltation if desired
  if(parametrize.qi == "all"){
    # parameterize integrated saltation mass concentration (kg m-2) or estimate it from LES data
    if(parametrize.int.salt.conc){
      # threshold friction velocity (m s-1)
      ustar_thresh = run1$boundary.flux$ustar_surf
      # ratio
      ustar_ratio = flx$u.star/ustar_thresh
      # Integrated mass flux in saltation (kg m-1 s-1) following Vionnet et al. (2014) based on Sorensen (2004)
      Mflux_int_salt = rho_sub * flx$u.star^3 / 9.81 * (1 - ustar_ratio^(-2)) * (2.6 + 2.5 * ustar_ratio^(-2) + 2 * ustar_ratio^(-1))
      # split integrated mass flux into velocity and integrated concentration components
      upart = 2.8 * ustar_thresh     # m s-1
      drift_conc = Mflux_int_salt / upart  # kg m-2
    } else{
      drift_conc = drift_conc_LES 
    }
    if(use_salt_levels){
      # drifting snow mass mixing ratio profile (kg kg-1) in saltation layer
      qi_final[1:sub_n_layers] = calc_q_bs_in_saltation(z = bs_Z[1:sub_n_layers], conc = drift_conc, ustar = flx$u.star, rho_air = rho_sub)
      sub_alpha  = alpha[1:sub_n_layers]
      # rate parameter of gamma distribution in saltation layer
      sub_lambda = sub_alpha / (2*meanR[1:sub_n_layers]) # m-1
      # number mixing ratio (kg-1) in saltation layer
      qni_final[1:sub_n_layers] = qi_final[1:sub_n_layers] * sub_lambda^3 * 6 / (pi*918.0) * gamma(sub_alpha)/gamma(sub_alpha+3)
    } else{
      # boundary conditions at h_salt
      q_salt = calc_q_bs_in_saltation(z = h_salt, conc = drift_conc, ustar = flx$u.star, rho_air = rho_sub)
      if(parametrize.alpha){
        alpha_salt = alpha.const
      } else{
        alpha_salt = approx(x = d.avg$z, y = alpha_LES, xout = h_salt, rule = 2)$y
      }
      lambda_salt = alpha_salt / dp.hsalt # m-1
      qn_salt     = q_salt* lambda_salt^3 * 6 / (pi*918.0) * gamma(alpha_salt)/gamma(alpha_salt+3)
    }
  }
  if(parametrize.qi != "no"){
    in_susp = (sub_n_layers+1):bs_Ntot
    n_susp = length(in_susp)
    # heights of full nodes for blowing snow in suspension
    z_full_bs = c(mean(c(h_salt, bs_Z[sub_n_layers+1])), bs_Z[(sub_n_layers+1):length(bs_dz)] + 0.5*bs_dz[(sub_n_layers+1):length(bs_dz)], z_top_boundary)
    if(ustar.Raupach != "no"){ # compute friction velocity profile according to Raupach (1991)
      my.ustar.bs = calc_ustar_raupach(z = z_full_bs, ustar = flx$u.star, ustar.t = ustar.s, h.salt = h_salt)
    } else{
      my.ustar.bs = rep(flx$u.star, length(z_full_bs))
    }
    # eddy diffusivity (m s-1) for blowing snow, defined at full levels of fine and coarse mesh
    K_eddy_bs = vk * my.ustar.bs * z_full_bs
    if(use_salt_levels){
      my_q_half = qi_final; my_qn_half = qni_final; my_alpha = alpha; my_temp = bs_temp; my_z_half = bs_Z; my_in_susp = in_susp
    } else{
      my_q_half = c(q_salt, qi_final); my_qn_half = c(qn_salt, qni_final); my_alpha = c(alpha_salt, alpha); my_temp = c(NA, bs_temp); 
      my_z_half = c(h_salt, bs_Z); my_in_susp = in_susp + 1
    }
    bs_change = bs_diffu_sedim_1D(q_half = my_q_half, qn_half = my_qn_half, alpha = my_alpha, temp = my_temp, z_half = my_z_half, in_susp = my_in_susp,
                                  z_full_susp = z_full_bs, K = K_eddy_bs, q_top = qi_top, qn_top = qni_top, alpha_top = alpha_ub,
                                  temp_top = BC_T_top, rho = rho_sub, sedim_diff = sedimentation_option)
    # update blowing snow mixing ratios (kg kg-1 or kg-1)
    qi_final[in_susp]  = qi_final[in_susp]  + dt_in * bs_change$q_change
    qni_final[in_susp] = qni_final[in_susp] + dt_in * bs_change$qn_change
    # print a warning if there are negative mixing ratios
    if(any(qi_final < 0 | qni_final < 0)) warning("Negative mixing ratio of blowing snow in domain.")
    # update mean radius per height if blowing snow mass mixing ratio is above threshold for which sublimation is calculated (otherwise not used)
    in.update.R = in_susp[qi_final[in_susp] >= 1e-10 & qni_final[in_susp] > 0]
    alpha_update = alpha[in.update.R]
    lambda_susp = ( pi*918.0/6.0 * qni_final[in.update.R]/qi_final[in.update.R] * gamma(alpha_update+3)/gamma(alpha_update) )**(1.0/3.0)
    meanR[in.update.R] = 0.5 * alpha_update / lambda_susp
  }
  
  ## drifting/blowing snow sublimation (kg kg-1 s-1), relative humidity (%), latent heat of sublimation (J kg-1), layer thickness (m), and saturation specific humidity (kg kg-1) per height
  if(modify.TM.below30cm){ # based on modified TM formula
    if(T.q.option == "linear_without_salt_layer") stop("The T.q.option linear_without_salt_layer is not prepared for modify.TM.below30cm = T.")
    # Assume here that first level is at z=7.5mm
    if(bs_Z[1] != 0.0075) warning(paste("Lowest level is expected to be z = 0.0075 m but it is z =", bs_Z[1], "m."))
    # indexes for height levels below 0.3 m
    in.30cm = which(bs_Z < 0.3)
    RH_1      = calc_RH_ice(Temp = bs_temp[1], q = bs_qv[1], rho = rho_sub)
    delta.T.1 = rep(bs_temp[1] - TSK, length(in.30cm))
    lm.term = coef_dTdt(z = bs_Z[in.30cm], delta.T = delta.T.1, RH = RH_1)
    lm.term = lapply(lm.term, function(x){ c(x, rep(0, bs_Ntot - length(in.30cm))) })
    lm.term$z_gap = mean(c(bs_Z[sub_n_layers]+0.5*dz_salt, bs_Z[sub_n_layers]+0.5*bs_dz[sub_n_layers])) # bs_Z[sub_n_layers]+0.5*bs_dz[sub_n_layers-1]
    # coefficients for the height in the upper part of the cell around the saltation layer height
    lm.term$gap = coef_dTdt(z = lm.term$z_gap, delta.T = delta.T.1[1], RH = RH_1)
    lm.term$rho_p = 918.0  # kg m-3
    lm.term$c_p_p = 2035.7 # J kg-1 K-1
  } else{ # based on original TM formula
    lm.term = NULL
  }
  # height (m) above which sublimation is calculated
  sub_z_exclude = ifelse( use_salt_levels, 0, mean( c(h_salt, Z_fine[1]) ) )
  result = calc_bs_subl(bs_Z = bs_Z, Z_boundary = c(sub_z_exclude, z_top_boundary), sub_n_layers = sub_n_layers, dz_salt = dz_salt, bs_temp = bs_temp,
                        bs_qv = bs_qv, p = press, RHO = RHO, alpha = alpha, qi_final = qi_final, bs_meanR = meanR, h_salt = h_salt, transient_term = lm.term)
  
  for(k in 1:bs_Ntot){
    if(result$bs_sublim_qi[k] <= 0){
      # If there is sublimation (negative value, particles lose mass), limit the absolute magnitude to the amount of blowing snow that is present
      if(-qi_final[k]/dt_in > result$bs_sublim_qi[k]){
        print(paste("k = ", k, ", t = ", t, "s: Limiting sublimation to the amount of blowing snow"))
      }
      result$bs_sublim_qi[k] = max(-qi_final[k]/dt_in, result$bs_sublim_qi[k]) #, (bs_qv[k]-result$bs_qsat[k])*0.999/dt_in)
    }
  }
  
  # specific humidity source due to drifting snow sublimation (kg kg-1 s-1)
  qv_src = result$bs_sublim_qi * (-1)
  # temperature/sensible heat source due to drifting snow sublimation (K s-1)
  t_src  = (-1) * ( result$bs_storage_change - result$bs_sublim_qi * result$lsub ) / Cp_air
  # T and q correction terms
  q_corr   = dt_in * qv_src # kg kg-1
  q_corr_i = q_corr # keep a copy for effect on blowing snow mixing ratios
  t_corr   = dt_in * t_src  # K
  if(T.q.option == "prognostic"){
    in.check = 1:bs_Ntot
  } else{
    in.check = (snc_N+1):bs_Ntot
    # contribution of saltation mesh and fine mesh to moisture (kg kg-1 m s-1) and heat fluxes (K m s-1)
    qw_flux_salt_fine = sum(qv_src[1:snc_N] * result$bs_grid_height[1:snc_N], na.rm = T)
    Tw_flux_salt_fine = sum(t_src[1:snc_N] * result$bs_grid_height[1:snc_N], na.rm = T)
    # scale T and q correction term of first coarse level
    q_corr[snc_N+1] = q_corr[snc_N+1] * result$bs_grid_height[snc_N+1] / (Z_top + dz_coarse/2)
    t_corr[snc_N+1] = t_corr[snc_N+1] * result$bs_grid_height[snc_N+1] / (Z_top + dz_coarse/2)
    if(feedback.on.surfFlux){ # if the sublimation effect of saltation mesh and fine mesh feeds back on surface fluxes
      BC_q_bottom[[1]] = BC_q_bottom[[1]] + qw_flux_salt_fine
      BC_T_bottom[[1]] = BC_T_bottom[[1]] + Tw_flux_salt_fine
    } else{ # if the sublimation effect of saltation mesh and fine mesh feeds back on first coarse layer
      # T and q sources of saltation and fine meshes scaled to apply them at first coarse layer
      q_src_scaled = qw_flux_salt_fine / (Z_top + dz_coarse/2) # (kg kg-1 s-1)
      t_src_scaled = Tw_flux_salt_fine / (Z_top + dz_coarse/2) # (K s-1)
      # Update T and q correction term of first coarse layer
      q_corr[snc_N+1] = q_corr[snc_N+1] + dt_in * q_src_scaled
      t_corr[snc_N+1] = t_corr[snc_N+1] + dt_in * t_src_scaled
    }
  }
  
  ## Compute explicitly T and q change due to vertical turbulent transport
  # Fluxes through upper boundary
  if(BC_T_top[[2]] == "DRC"){
    T_flux_top  = -K_eddy[length(K_eddy)] * (BC_T_top[[1]] - bs_temp[length(bs_temp)]) / (dz_coarse/2) # K m s-1
    my_BC_T_top = list(T_flux_top, "FLX")
  } else{
    my_BC_T_top = BC_T_top
  }
  if(BC_q_top[[2]] == "DRC"){
    q_flux_top  = -K_eddy[length(K_eddy)] * (BC_q_top[[1]] - bs_qv[length(bs_qv)]) / (dz_coarse/2) # kg kg-1 m s-1
    my_BC_q_top = list(q_flux_top, "FLX")
  } else{
    my_BC_q_top = BC_q_top
  }
  # Effect of vertical turbulent transport
  if(T.q.option == "prognostic"){
    T_mixing = solve_diffusion(x = bs_temp, dz_half = diff(bs_Z), dz_full = diff(z_full_K), K = K_eddy, BC1 = BC_T_bottom, BC2 = my_BC_T_top)
    q_mixing = solve_diffusion(x = bs_qv,   dz_half = diff(bs_Z), dz_full = diff(z_full_K), K = K_eddy, BC1 = BC_q_bottom, BC2 = my_BC_q_top)
  } else{ # only on coarse mesh
    T_mixing = solve_diffusion(x = bs_temp[(snc_N+1):bs_Ntot], dz_half = dz_coarse, K = K_eddy, BC1 = BC_T_bottom, BC2 = my_BC_T_top)
    q_mixing = solve_diffusion(x = bs_qv[(snc_N+1):bs_Ntot],   dz_half = dz_coarse, K = K_eddy, BC1 = BC_q_bottom, BC2 = my_BC_q_top)
  }
  dTdt_mix = T_mixing$change
  dqdt_mix = q_mixing$change
  
  # If the q correction terms remove or add more moisture than the amount that causes saturation, print warning message
  for(k in in.check){
    # expected specific humidity resulting from vertical mixing only
    q_tmp_mix = bs_qv[k] + dt_in * dqdt_mix[ match(k, in.check) ]
    # expected specific humidity also taking into account the exchange with drifting/blowing snow
    q_tmp     = q_tmp_mix + q_corr[k]
    # If there is blowing snow sublimation and q would become higher than current saturation value, print warning at first occasion
    if(q_corr[k] > 0 & q_tmp > (result$bs_qsat[k])){
      if(flag.warn$sublimation) warning( paste("k = ", k, ", t = ", round(t,digits = 3), "s: Exceeding RH of 100 %", sep = "") )
      flag.warn$sublimation = F
    } else{
      # If we use original TM formula and there is vapour deposition on blowing snow and q would become lower than current saturation value, then print warning message once
      if( (!modify.TM.below30cm | k > (sub_n_layers)) & q_corr[k] < 0 & q_tmp < (result$bs_qsat[k]) ){
        if(flag.warn$deposition) warning( paste("k = ", k, ", t = ", round(t,digits = 3), "s: Deposition leads to RH < 100 %", sep = "") )
        flag.warn$deposition = F
      }
    }
  }
  # if blowing snow is parametrized, vapour exchange affects blowing snow mixing ratios
  if(parametrize.qi != "no"){
    # change in number mixing ratio (kg-1)
    qn_corr_i = rep(NA, bs_Ntot)
    for(k in in_susp){
      if(qi_final[k] > 0){ # if blowing snow is present
        if(q_corr_i[k] > 0){ # if there is sublimation, keep the same ratio of number mixing ratio to mass mixing ratio of particles
          qn_corr_i[k] = q_corr_i[k] * qni_final[k]/qi_final[k]
        } else{ # if there is vapor deposition, keep the number mixing ratio of particles unchanged
          qn_corr_i[k] = 0
        }
        if(qn_corr_i[k] > qni_final[k]){
          qn_corr_i[k] = qni_final[k]
          warning(paste("k =", k, "Reducing bs number mixing ratio to avoid negative mixing ratio."))
        }
      } else{ # if blowing snow is absent
        qn_corr_i[k] = 0
      }
    }
    # correct blowing snow mixing ratios for sublimation effect
    qi_final[in_susp]  = qi_final[in_susp]  - q_corr_i[in_susp]
    qni_final[in_susp] = qni_final[in_susp] - qn_corr_i[in_susp]
    # total mass and number loss of particles considering fine and coarse levels (z > approx. 0.3 m) (kg kg-1 m s-1 and kg-1 m s-1, resp.) # only for output
    m_loss = sum(q_corr_i[in_susp] * result$bs_grid_height[in_susp] / dt_in)
    N_loss = sum(qn_corr_i[in_susp] * result$bs_grid_height[in_susp] / dt_in)
  }
  
  ## Store output
  if(i %% as.integer(interval.out/dt_in) == 0){
    in.out = as.integer(round(t/interval.out))
    # If latent/sensible heat exchange was reduced due to (under)saturation, this would not be reflected in the output (If we outputted q_corr/dt_in instead of q_src, we would count twice the exchange below the coarse mesh with the MO-bulk option)
    out.matrix = cbind(qv_src = qv_src, t_src = t_src, bs_qv, bs_temp, RH = result$RH, lsub = result$lsub)
    out[in.out,,]   = out.matrix
    out.bound[in.out,1:4] = c( q_mixing$flux_BC1, T_mixing$flux_BC1, q_mixing$flux_BC2, T_mixing$flux_BC2)
    if(parametrize.qi != "no"){
      out.qi[in.out,]  = qi_final
      out.qni[in.out,] = qni_final
      out.bound[in.out,5:14] = c(bs_change$q_flx_diffu_BC1, bs_change$q_flx_diffu_BC2, bs_change$q_flx_sedim_BC1, bs_change$q_flx_sedim_BC2, 
                                 bs_change$qn_flx_diffu_BC1, bs_change$qn_flx_diffu_BC2, bs_change$qn_flx_sedim_BC1, bs_change$qn_flx_sedim_BC2, 
                                 m_loss, N_loss)
    }
  }
  
  ## Update T, q, RHO
  bs_temp[in.check] = bs_temp[in.check] + ( dt_in * dTdt_mix + t_corr[in.check] )
  bs_qv[in.check]   = bs_qv[in.check]   + ( dt_in * dqdt_mix + q_corr[in.check] )
} # end of time loop

dimnames(out) = list(time     = as.character(seq(interval.out, by=interval.out, length.out=dim(out)[1])), 
                     z        = as.character(round(bs_Z,4)), 
                     variable = colnames(out.matrix))
names(out.bound)[1:4] = c("lb_qw_flux","lb_Tw_flux", "ub_qw_flux","ub_Tw_flux")
if(parametrize.qi != "no"){
  names(out.bound)[5:14] = c(names(bs_change)[-c(1,2)], "m_loss_sublim", "N_loss_sublim")
} else{
  out.qi[1,]  = qi_final
  out.qni[1,] = result$qni
}

# save output
bs_grid_height = result$bs_grid_height
TM.out = ifelse(modify.TM.below30cm, "_transient-Tp-subset", "")
dp.out = if(parametrize.d){ ifelse(dp.hsalt==0.0002, "", paste("_dp-salt",dp.hsalt*1e6,sep="")) }else{ "_dp-from-LES" }
if(parametrize.qi=="all"){       qi.out = "_q-bs-param-all"}
if(parametrize.qi=="suspension"){qi.out = "_q-bs-param-susp"}
if(parametrize.qi=="no"){        qi.out = ""}
UBC.out= ifelse(f.shift.BC.top == 0, "", "_shiftUBC")
stab.out = ifelse(corr.stability, "", "_neutral")
z0.out = ifelse(default.z0, "_default-z0", "")
level.out = if(use_salt_levels){ ifelse(only1salt.level, "_1salt-level", "") }else{ "_no-salt-level" }
thresh.out = if(parametrize.qi != "no"){ "_consistent-thresh" }else{ "" } # ifelse(high.ustar.thresh, "_high-thresh", "_low-thresh")
sedim.out = if(parametrize.qi != "no"){ if(sedimentation_option=="forward"){ "_sedim-forward" }else{ if(sedimentation_option=="q_forward_v_backward"){ "_sedim-v-backward" }else{ if(sedimentation_option=="central"){ "_sedim-central"}else{ "" } } } }else{ "" }
alpha.out = if(parametrize.alpha){ ifelse(alpha.const != 3, paste("_alpha",alpha, sep=""), "") }else{ "_alpha-from-LES" }
spacing.out = if(log_spacing_fine){ "_log-spacing" }else{ "" }
if(!(Z_fine[1] %in% c(0.2,0.5))) spacing.out = paste(spacing.out, "_with_", round(Z_fine[1]*100), "cm", sep="")
if(length(Z_fine) > 10) spacing.out = paste(spacing.out, "_high-res", sep = "")
hsalt.out = if(is.numeric(parametrize.h_salt)){ paste("_hsalt",round(100*parametrize.h_salt),"cm",sep="") }else{ "" }
ustar.out = ifelse(impose.ustar, "_ustar-imp", "")
Raupach.out = ifelse(ustar.Raupach == "no", "", paste("_Raupach-", ustar.Raupach, sep=""))
salt.conc.out = if(parametrize.qi == "all"){ ifelse(parametrize.int.salt.conc, "", "_IntSaltConc-from-LES") }else{ "" }
fn.out = paste("Parametr_output_",case,"_",T.q.option,TM.out,dp.out,qi.out,UBC.out,stab.out,z0.out,level.out,thresh.out,sedim.out,alpha.out,spacing.out,hsalt.out,ustar.out,Raupach.out,salt.conc.out,".Rdata", sep = "")
# fn.out = file.path(paste("Parametr_output_",case,sep=""), fn.out)
# save model output
if(parametrize.qi != "no"){
  if(use_salt_levels){
    save(out, out.bound, bs_grid_height, out.qi, out.qni, meanR, file = fn.out)
  } else{
    save(out, out.bound, bs_grid_height, out.qi, out.qni, meanR, h_salt, q_salt, qn_salt, alpha_salt, lambda_salt, file = fn.out)
  }
} else{
  save(out, out.bound, bs_grid_height, out.qi, out.qni, meanR, file = fn.out)
}

