# Script for reading and processing LES-LSM output files

# Armin Sigmund

# Settings ----------------------------------------------------------------

rm(list=ls())
# working directory: location of the Postprocessing_code folder
setwd("~/Documents/PhD_work/my_paper_drafts/paper_parametrization/data_for_EnviDat/Postprocessing_code/")
my.cases = c("1","2","3a","3b")
case      = "1"
in.case   = match(x = case, table = my.cases)
# directory of simulation (input and output) files
my.paths = file.path("~/Documents/PhD_work/LES_output_postprocessing/snowdrift_antarctica", 
                     c("S17_190111_0045/deep_domain/case1_dx10cm/",
                       "S17_190112_1105/case2_dx10cm/",
                       "S17_190111_2145/case3a_dx10cm/",
                       "S17_190111_2145/case3b_dx10cm/"))
path      = my.paths[in.case]
# filenames for processed data (to be saved)
fn.save = paste("processed_profiles_", c("case1_dx10cm_350-750s.Rdata",
                                         "case2_dx10cm_350-750s.Rdata",
                                         "case3a_dx10cm_350-750s.Rdata",
                                         "case3b_dx10cm_350-750s.Rdata"), sep = "")
fn.out = fn.save[in.case]
# path to functions
path.func = "my_functions/"

# specify HAVG_CNT3 in case it is not included in the PARAMETERS.py_0 file (depending on code version)
HAVG_CNT3 = F
# get parameters for the simulation from PARAMETERS.py_0
STD = "STD"
WALL_LAW = "WALL_LAW"
DRC = "DRC"
NEU = "NEU"
source(file = file.path(path, "PARAMETERS.py_0"))
source(file = file.path(path.func,"func_MO_bulk_fluxes.R"))
source(file = file.path(path.func,"func_Lsubl.R"))
source(file = "config_constants.R")
# correct NSTEPS
NSTEPS = 7.5e6
# other parameters: normalization factor for temperature and air density (kg/m^3)
if(case %in% c("3a","3b")){ t_scale = 268.71; rho_air = 1.18; z0_t = LBC_DZ0*0.3; z0_q = LBC_DZ0*0.4; source(file = "config_measured_values_S17_190111_2145.R") } else{
  # Case 2: S17
  if(case == "2"){   t_scale = 270.38; rho_air = 1.18; z0_t = LBC_DZ0*0.5; z0_q = LBC_DZ0*0.6; source(file = "config_measured_values_S17_190112_1105.R") } else{
    # Case 1: S17 Drift simulations
    if(case == "1"){ t_scale = 267.52; rho_air = 1.18; z0_t = LBC_DZ0*0.5; z0_q = LBC_DZ0*0.5; source(file = "config_measured_values_S17_190111_0045.R") }
  }
}
nu_molec= 1.24036021044e-5 # m^2/s^1 # molecular viscosity of air
#
#install.packages('zoo')
library(zoo)
library(fitdistrplus)
#
source(file = file.path(path.func, "func_read_binary_LES_output.R"))
source(file = file.path(path.func, "func_calc_es_les_RH_ice.R"))
source(file = file.path(path.func, "func_extract_profile_from_particles.R"))
source(file = file.path(path.func, "func_get_alpha.R"))
# text files to be read
fn = c("ustar","qstar","tstar","t_sgsFluxUpperBoundary","q_sgsFluxUpperBoundary")
# variables to be read from binary files (ta1 field), as named in the filename
varnames = c("u","w","t","q") 
# Explanations for simulation output:
  # q: specific humidity, kg kg^-1
  # t: air temperature, non-dimensional (K t_scale^-1)
  # u: wind velocity in x direction, m s^-1
  # w: vertical wind velocity, m s^-1
  # lsm_q: source/sink in specific humidity, kg kg^-1 s^-1
  # lsm_theta: total source/sink in temperature (lsm_theta_1 + lsm_theta_2), s^-1 (based on non-dimensionalized temperature)
  # lsm_theta_1: temperature source/sink due to sensible heat exchange with particles, s^-1
  # lsm_theta_2: temperature source/sink which brings newly sublimated/deposited vapor to the temperature of the resulting phase
  # qw: average product of instantaneous q and w, kg kg^-1 m s^-1, vertical moisture flux including both advection (should be negligible when averaging over long enough period) and turbulent transport
  # tw: average product of instantaneous t and w, m s^-1 (based on non-dimensionalized temperature), positive = upward!
  # tqz: vertical subgrid-scale humidity flux, kg kg^-1 m s^-1, positive = downward!
  # ttz: vertical subgrid-scale sensible heat flux, m s^-1 (based on non-dimensionalized temperature), positive = downward!
  # tzz: zz-component of the subgrid-scale stress tensor (tau)
  # c_hflux: horizontal particle mass flux, kg m^-2 s^-1, consistent with particles_* files
  # hflux: the extrapolated mean particle velocity u_p, m s^-1, does not follow the wind profile but it is very low (< 0.3 m s^-1) above a height of 0.5 m, likely because there are grid points without particles (value of zero) which reduce the horizontal and temporal average.
  # c_vflux: vertical particle mass flux, kg m^-2 s^-1
  # conc: particle mass concentration, kg m^-3
  # force_lsm_x, force_lsm_y, force_lsm_z: particle forcing on momentum, kg(particles) kg^-1(air) m s^-2
# anut: turbulent viscosity     (on w nodes, except at the lowest level where it refers to the uvp node)
# cs:   Smagorinsky coefficient (on w nodes, except at the lowest level where it refers to the uvp node)

# number of averaged fields in one binary file, can be overestimated (number of times)
n.fields = ceiling( NSTEPS / P_CNT3 )
# time at the beginning of the run (s)
t.start  = 0 # 400
# estimated beginning of quasi-stationary T and q conditions (s)
t.start.stnry = 350

# particles files to be read 
files.particle = list.files(path = file.path(path, 'output_0/lsm/'), pattern = "particles_", full.names = T)
# Either all particle files ...
# files.particle = list(tp1 = files.particle)
# ... OR: use particle files in given time intervals
fn.length  = nchar(files.particle)
fn.len     = unique(fn.length)
t.particle = fn.length
t.particle[which(fn.length==fn.len[1])] = as.integer(substr(files.particle[which(fn.length==fn.len[1])], start = fn.len[1]-11, stop = fn.len[1]-4))
t.particle[which(fn.length==fn.len[2])] = as.integer(substr(files.particle[which(fn.length==fn.len[2])], start = fn.len[2]-11, stop = fn.len[2]-4))
files.particle = list(tp1 = files.particle[t.particle > t.start.stnry/DT]) # & t.particle <= 450/DT])
# variables in the particles_* files
varnames.particle = c("jt_total","p_ip","x_p","y_p","z_p","u_p","v_p","w_p","T_p","d_p",
                      "re_p","nu_p","sh_p","mass_p","delta_m","delta_r","delta_T","timescale_stokes")
# initialization height of particles (m)
zp_init = 4*D_M

# surface temperature (degC)
t0 = DLBC_SC1*t_scale-273.15
# surface specific humidity (g kg-1)                                        #
q0 = DLBC_SC2 * 1000

# Is there a header line in the mass_balance.txt file?
tf.header = T


# read and process boundary fluxes (ustar, tstar, qstar) -------------------------------------
source(file = "read_LES_boundary_fluxes.R")

# plot time series of boundary fluxes
source(file = "plot_LES_boundary_fluxes.R")

# TKE (m2 s-2), averaged over all grid points
kin.energy = read.table(file = file.path(path, "ke.txt"))
kin.energy = zoo(kin.energy, seq(t.start + WBASE * DT, by = WBASE * DT, length.out = nrow(kin.energy)))
plot(kin.energy, xlab = "Time (s)", ylab = expression(TKE~(m^2~s^{-2})))
if(LSM){
  t.init = LSM_INIT*DT
  abline(v=t.init, lty = 2, col = "darkgrey")
}

# read ta1 field ----------------------------------------------------------
source("read_process_ta1_fields.R")
for(i in 1:length(avg)){
  colnames(avg[[i]]) = round(z, digits = 5)
}
# save horizontally averaged vertical profiles
save(avg, file = paste("ta1_vertical_profiles_case_",case,".Rdata", sep = ""))
# relative humidity (%)
rh = calc_RH_ice(avg$t+273.15, avg$q/1000, rho_air)

# read particles files/data -----------------------------------------------
ptcl = lapply(files.particle, function(x){
  d = lapply(x, scan)
  d = do.call(c, d)
  d = matrix(d, ncol = 18, byrow = T)
  d = as.data.frame( d )
  names(d) = varnames.particle
  # number of particles per parcel
  d$ppp = d$mass_p / (P_RHO*pi*d$d_p^3 / 6)
  return(d)
})

# times (s) of particles output
time.ptcl = sort(unique(ptcl$tp1$jt_total))*DT
range(time.ptcl)
# heights of upper boundaries between grid layers (lowest layer covers entrainment height, to be excluded later for some variables such as particle mass flux)
bound = c(zp_init, z.w[-1])
# process particles data
source(file = "calc_particles_profiles.R")


# Entrainment/deposition data from mass_balance.txt file ------------------------------------------
source(file = "read_plot_mass_balance_file.R")


# time-averaged ta1 profiles for quasi-stationary period ------------------------------------------------------------------

run1 = list(ta1 = sapply(avg, function(x){
  apply( window(x, start = I(t.start.stnry), end = I(NSTEPS*DT)), 2, mean )
}))
run1$ta1 = data.frame( z, run1$ta1, rh = apply(window(rh,start = t.start.stnry, end = NSTEPS*DT), 2, mean) )
head(run1$ta1)


# Fit gamma distributions ---------------------------------------------------

# shape parameter inferred from mass and number mixing ratios and mean diameter
alpha_est = get_alpha(qi = mass.conc / rho_air, qni = number.conc / rho_air, dp = d_p_mean)
# inferred rate parameter of gamma distribution
lambda_est = alpha_est / d_p_mean # m-1
my.alpha  = rep(NA, length(z))
my.lambda = rep(NA, length(z))
for(i in 1:length(z)){
  print(i)
  my.z = c(bound[1]+0.5*(bound[2]-bound[1]), z[-1])[i]
  in.z = which( ptcl$tp1$z_p > z.w[i] & ptcl$tp1$z_p < z.w[i+1] )
  # convert to micrometer because fitting algorithm does not work if values are very small
  d_p_i = ptcl$tp1$d_p[in.z] * 1e6
  ppp_i = round(ptcl$tp1$ppp[in.z])
  my.start = list(shape=alpha_est[i], rate=lambda_est[i]/1e6)
  my.fit = fitdist(d_p_i, distr = "gamma", keepdata = F, weights = ppp_i, start = my.start)
  my.alpha[i]  = coef(my.fit)[1]
  my.lambda[i] = coef(my.fit)[2]*1e6 # m-1
}


# add processed particles data ------------------------------------------------------------------------
run1$ptcl = data.frame( z = c(bound[1]+0.5*(bound[2]-bound[1]), z[-1]),
                        mass_hflux,
                        number_hflux,
                        mass.conc,
                        number.conc,
                        d_p_mean,
                        d_p_sigma,
                        u_p_mean,
                        #w_p_mean,
                        T_p_mean,
                        FQ_p,
                        Ft_p = Ft_p_1 + Ft_p_2,
                        Ft_p_1,
                        Ft_p_2,
                        q.p.minus.air = q.diff.layer,
                        t.p.minus.air = t.diff.layer,
                        gamma.shape = my.alpha,
                        gamma.rate  = my.lambda)
run1$ptcl.with.time  = list(time = time.ptcl,
                            mass.conc.with.time = mass.conc.with.time,
                            hflux_with_time = hflux_with_time,
                            number_hflux_with_time = number_hflux_with_time,
                            u_p_mean_with_time = u_p_mean_with_time,
                            FQ_p_with_time = FQ_p_with_time,
                            S.latent = S.latent,
                            Ft_p_1_with_time = Ft_p_1_with_time,
                            Ft_p_2_with_time = Ft_p_2_with_time,
                            T_p_mean_with_time = T_p_mean_with_time)
run1$ptcl.int = data.frame(time = time.ptcl, transport, exchange.latent)
# convert units of sensible heat source from K s-1 to W m-3
run1$ptcl.with.time$S.sensible = (run1$ptcl.with.time$Ft_p_1_with_time + run1$ptcl.with.time$Ft_p_2_with_time) * rho_air * c_p_a
# vertically integrate the sensible heat source (W m-2)
run1$ptcl.int$exchange.sensible = apply(run1$ptcl.with.time$S.sensible, 1, function(x){ sum( x * diff(bound) ) })


# add average heat fluxes (W/m^2) at the lower and upper boundaries ----------------------------
# ... and surface friction velocity (m/s)
run1$boundary.flux = boundary.flux.avg


# add heights of w nodes --------------------------------------------------
run1$z.w = z.w


# save profiles together with units ------------------------------------------------------------
labels = list(z = "z (m)", q = expression(q~(g~kg^{-1})), t = expression("T"~(phantom()^degree*C)), u = expression(u~(m~s^{-1})),
              w = expression(u~(m~s^{-1})), rh = "RH (%)", 
              mass_hflux = expression(Flux~(kg~m^{-2}~min^{-1})), number_hflux = expression(F[N]~(cm^{-2}~s^{-1})), mass.conc = expression("Mass concentration"~(kg~m^{-3})),
              number.conc = expression("Number concentration"~(m^{-3})), d_p_mean = expression(d[p]~("m")),
              u_p_mean = expression(u[p]~(m~s^{-1})), T_p_mean = expression("T"[p]~(phantom()^degree*C)), FQ_p = expression(S[q]~(g~kg^{-1}~s^{-1})),
              Ft_p = expression(S[h]~(K~s^{-1})), Ft_p_1 = expression(S[h]~(K~s^{-1})), Ft_p_2 = expression(S[h]~(K~s^{-1})),
              q.p.minus.air = expression(q[p]-q[a]~(g~kg^{-1})), t.p.minus.air = expression("T"[p]-"T"[a]~(K)),
              gamma.shape = expression(alpha~(1)), gamma.rate = expression(lambda~(m^{-1})), d_p_sigma = expression(sigma[d[p]]~("m")),
              time = "Time (s)", S.latent = expression(S[lat]~(W~m^{-3})), S.sensible = expression(S[sens]~(W~m^{-3})),
              transport = expression(Transport~(kg~m^{-1}~min^{-1})), exchange.latent = expression(LE[bs]~(W~m^{-2})), 
              exchange.sensible = expression(H[bs]~(W~m^{-2})), qh_surf = expression(H[surf]~(W~m^{-2})), ql_surf = expression(LE[surf]~(W~m^{-2})), 
              qh_ub = expression(H[top]~(W~m^{-2})), ql_ub = expression(LE[top]~(W~m^{-2})), ustar_surf = expression(u["*0"]~(m~s^{-1})),
              z.w = expression(z[w]~(m)) )

save(run1, labels, file = fn.out)


# Save data for publication (Envidat) ---------------------------

## particles data

# sensible heat exchange (J s-1)
ptcl[[1]]$q_sensible = ptcl[[1]]$q_conv + ptcl[[1]]$q_evap
# save selected properties of each parcel
in.save = match(c("jt_total","z_p","u_p","T_p","d_p","mass_p","delta_m","delta_T","q_sensible"), names(ptcl[[1]]))
ptcl = ptcl[[1]][, in.save]
gc()
# convert units of particle temperature to degC
ptcl$T_p = ptcl$T_p * t_scale - 273.15
# convert units of change in particle temperature to K s-1
ptcl$delta_T = ptcl$delta_T * t_scale
# specify time (s)
ptcl$jt_total = ptcl$jt_total * DT
colnames(ptcl)[1] = "time"
save(ptcl, file = paste("LES-LM_particles_case_",case,".Rdata", sep = ""), compression_level = 9)

## fluxes at lower and upper boundaries

d.out = data.frame(time(xstar$ustar),      # s
                   xstar$ustar,            # m s-1
                   xstar$qstar * 1000,     # g kg-1
                   xstar$tstar * t_scale,  # K
                   xstar$q_sgsFluxUpperBoundary * 1000,    # g kg-1 m s-1
                   xstar$t_sgsFluxUpperBoundary * t_scale) # K m s-1
colnames(d.out) = c("Time","u_star_surf", "q_star_surf", "t_star_surf", "q_flux_top", "t_flux_top")
fn.boundary = paste("boundary_fluxes_case_",case,".csv", sep="")
write.csv(round(d.out, digits = 6), file = fn.boundary, row.names = F)
