# Script for plotting and comparing parametrization results

# Armin Sigmund

# Settings ----------------------------------------------------------------
rm(list=ls())
# working directory
setwd("~/Documents/PhD_work/my_paper_drafts/paper_parametrization/data_for_EnviDat/Postprocessing_code/")

# path to functions
path.func = "my_functions/"
source(file = file.path(path.func,"func_Lsubl.R"))
source(file = file.path(path.func,"func_MO_bulk_fluxes.R"))
source(file = file.path(path.func,"func_calc_es_les_RH_ice.R"))
source(file = file.path(path.func,"func_calc_h_salt.R"))
# get constants like gas constants and specific heat capacity of air
source(file = "config_constants.R")

# assume constant air density (kg m-3) like in LES
rho_sub = 1.18

path.LES = "../LES-LM_main_results/"
file.forcings.LES.mesh = c(file.path(path.LES, "processed_profiles_case1_dx10cm_350-750s.Rdata"),
                           file.path(path.LES, "processed_profiles_case2_dx10cm_350-750s.Rdata"),
                           file.path(path.LES, "processed_profiles_case3a_dx10cm_350-750s.Rdata"),
                           file.path(path.LES, "processed_profiles_case3b_dx10cm_350-750s.Rdata"))
# path to measurements for the 4 cases
fn.meas = list(Case1  = "config_measured_values_S17_190111_0045.R",
               Case2  = "config_measured_values_S17_190112_1105.R",
               Case3a = "config_measured_values_S17_190111_2145.R",
               Case3b = "config_measured_values_S17_190111_2145.R")

case = c("case1_dx10cm", "case2_dx10cm", "case3a_dx10cm", "case3b_dx10cm")

my.option = c("MO-bulk_dp-from-LES_no-salt-level_alpha-from-LES_ustar-imp",
              "MO-bulk_dp-from-LES_alpha-from-LES_log-spacing_with_15cm_ustar-imp_Raupach-LES-based",
              "prognostic_dp-from-LES_alpha-from-LES_log-spacing_with_15cm_ustar-imp_Raupach-LES-based",
              "prognostic_transient-Tp-subset_dp-from-LES_alpha-from-LES_log-spacing_with_15cm_ustar-imp_Raupach-LES-based",
              "prognostic_transient-Tp-subset_dp-from-LES_1salt-level_alpha-from-LES_log-spacing_with_15cm_ustar-imp_Raupach-LES-based") #"MO-bulk_transient-Tp",
# relative path specifying where output of 1D model is stored
path.1D.output = rep( "../1D_model_code/", length(case) )
# path.1D.output = sapply(case, function(i){ file.path( "~/Documents/PhD_work/CRYOWRF_1D/sublim_in_saltation_tests", paste("Parametr_output_",i,sep="") ) })
# files with 1D model (parametrization) results
fn       = lapply(1:length(case), function(i){ paste("Parametr_output_",case[i],"_",my.option,".Rdata", sep = "") })
path.par = lapply(1:length(case), function(i){ file.path(path.1D.output[i], fn[[i]]) })

# paths for set-ups with modeled blowing snow in suspension layer (or saltation + suspension layer)
z_h_salt_modified  = c("_with_5cm_high-res_hsalt4", "_with_5cm_high-res_hsalt4", "_with_8cm_high-res_hsalt7", "_with_8cm_high-res_hsalt7")
## model set-ups with modeled particle mixing ratios
my.bs.option = list(   rep("prognostic_transient-Tp-subset_dp-from-LES_q-bs-param-susp_consistent-thresh_sedim-v-backward_log-spacing_with_15cm_ustar-imp_Raupach-LES-based.Rdata", 4),
                       paste("prognostic_transient-Tp-subset_dp-from-LES_q-bs-param-susp_consistent-thresh_sedim-central_log-spacing", z_h_salt_modified,"cm_ustar-imp_Raupach-LES-based.Rdata", sep = ""),
                         rep("prognostic_transient-Tp-subset_q-bs-param-all_consistent-thresh_sedim-v-backward_log-spacing_with_15cm_ustar-imp_Raupach-LES-based.Rdata", 4),
                         rep("MO-bulk_q-bs-param-all_no-salt-level_consistent-thresh_sedim-forward_ustar-imp.Rdata", 4),
                         rep("prognostic_transient-Tp-subset_dp-from-LES_q-bs-param-susp_consistent-thresh_sedim-v-backward_log-spacing_with_15cm_hsalt13cm_ustar-imp_Raupach-LES-based.Rdata", 4),
                         rep("prognostic_transient-Tp-subset_dp-from-LES_q-bs-param-susp_consistent-thresh_sedim-central_log-spacing_with_14cm_high-res_hsalt13cm_ustar-imp_Raupach-LES-based.Rdata", 4),
                         rep("prognostic_transient-Tp-subset_dp-salt140_q-bs-param-all_consistent-thresh_sedim-v-backward_log-spacing_with_15cm_hsalt13cm_ustar-imp_Raupach-LES-based.Rdata", 4),
                         rep("MO-bulk_dp-salt140_q-bs-param-all_no-salt-level_consistent-thresh_sedim-forward_hsalt13cm_ustar-imp.Rdata", 4) )
  parametrize.alpha = rep(T, 8)

path.par.bs = lapply(1:length(my.bs.option[[1]]), function(i){ 
  sapply(1:length(my.bs.option), function(j){
    list.files(path = path.1D.output[i], pattern = my.bs.option[[j]][i], full.names = T) 
  })
})

# surface temperatures (degC)
t0 = c(267.52, 270.38, 268.71, 268.71) -273.15
q0 = c(2.61312855682389, 3.296502626, 2.880118 , 2.880118)
RH0 = calc_RH_ice(Temp = t0+273.15, q = q0/1000, rho = rho_sub)
# UBC in 1D model
t9m = c(-4.093, -3.630, -4.658, -3.730) # degC
q9m = c(2.502, 3.028, 2.824, 3.040) # g kg-1
RH9m = calc_RH_ice(Temp = t9m+273.15, q = q9m/1000, rho = rho_sub)
# roughness lengths (m)
rough_len_m_LES = rep(1e-4, 4)
rough_len_t_LES = c(5e-5, 5e-5, 3e-5, 3e-5)
rough_len_q_LES = c(5e-5, 6e-5, 4e-5, 4e-5)


# time index to be plotted
k = 100
# for parametrizations 1 to 4 in plot
my.cols = c("orange","cyan2",2,"purple",4)
my.pch = c(20,15,1,17,6)

# logical for writing data of plots to csv file (for EnviDat repository)
write2csv = F #T
case.label = c("1","2","3a","3b")

# get measured data -------------------------------------------------------------------------------
my.meas = list()
my.unc = list()
for(i in 1:length(fn.meas)){
  source(file = fn.meas[[i]])
  my.meas[[i]]    = meas
  my.meas[[i]]$LE = ql.meas
  my.meas[[i]]$H  = qh.meas
  my.unc[[i]]     = unc
}
names(my.meas) = names(my.unc) = names(fn.meas)
q0.meas = sapply(my.meas, function(i){ i$q[1] })

# Load and post-process parametrization and LES results --------------------------------------------
my.d = list()
my.run1 = list()
for (i in 1:length(fn)){
  my.d[[i]] = list()
  for (j in 1:length(my.option)){
    load(file = path.par[[i]][j])
    my.d[[i]][[j]] = list(out=out, out.bound=out.bound, out.qi=out.qi, out.qni=out.qni, bs_grid_height=bs_grid_height, meanR=meanR)
  }
  names(my.d[[i]]) = my.option
  # load LES data on LES mesh (for plotting blowing snow profile)
  load(file = file.forcings.LES.mesh[i])
  my.run1[[i]] = run1
}
names(my.d)     = case
names(my.run1) = case

# remove unnecessary variables
rm(out, out.bound, run1, bs_grid_height)

# column index
in.qv_src = match("qv_src", dimnames(my.d[[1]][[1]]$out)$variable)
in.t_src  = match("t_src", dimnames(my.d[[1]][[1]]$out)$variable)
in.t = match("bs_temp", dimnames(my.d[[1]][[1]]$out)$variable)
in.qv= match("bs_qv", dimnames(my.d[[1]][[1]]$out)$variable)
in.RH= match("RH", dimnames(my.d[[1]][[1]]$out)$variable)
in.lsub = match("lsub", dimnames(my.d[[1]][[1]]$out)$variable)

# index of a 1D model run without the 1salt-level and no-salt-level options
my_index = ( 1:length(my.option) )[-c(grep("1salt-level", my.option), grep("no-salt-level", my.option))][1]
# height of saltation layer (m)
my_h_salt = lapply(my.d, function(i){ sum( i[[my_index]]$bs_grid_height < 0.0151 ) * 0.015 + 0.0075 })

Z_top = 4.5 * 0.5 # corresponds to the first WRF U-level?!
Z_fine_linear = c(0.5, 0.86, 1.21, 1.57, 1.92) # c(0.5, 1.35)
# Z_fine = c(0.2, 0.33, 0.55, 0.92, 1.53) 
# fine levels with log-spacing starting at 0.15 m
Z_fine_with15cm = c(0.15, 0.23, 0.35, 0.54, 0.83, 1.27, 1.96)
# heights of fine mesh and saltation mesh
snc_Z = lapply(1:length(my_h_salt), function(i){ 
  lapply(1:length(my.d[[i]]), function(j){
    if(grepl("1salt-level", path.par[[i]][j])){
      c(0.0075, Z_fine_with15cm)
    } else{
      if(grepl("log-spacing", path.par[[i]][j])){
        c( my.run1[[i]]$ta1$z[which(my.run1[[i]]$ta1$z < (my_h_salt[[i]]+0.0075))], Z_fine_with15cm )
      } else{
        Z_fine_linear
      }
    }
  })
})
# number of layers in fine mesh and saltation mesh
snc_N = lapply(snc_Z, function(i){ 
  lapply(i, length)
})
# heights of coarse mesh
Z_coarse = seq(Z_top, 9, 4.5)
# all heights
bs_Z = lapply(snc_Z, function(i){ 
  lapply(i, function(j){
    c(j, Z_coarse) 
  })
})
bs_Ntot = lapply(bs_Z, function(i){ 
  lapply(i, length)
})
# number of layers in saltation
sub_n_layers = lapply(1:length(bs_Z), function(i){ 
  lapply(1:length(bs_Z[[i]]), function(j){
    sum(bs_Z[[i]][[j]] < (my_h_salt[[i]]+0.0075))
  })
})
# heights (m) of half nodes in suspension layer
z_susp = lapply(1:length(bs_Z), function(i){ 
  lapply(1:length(bs_Z[[i]]), function(j){
    bs_Z[[i]][[j]][(sub_n_layers[[i]][[j]]+1):length(bs_Z[[i]][[j]])]
  })
})
# height differences
bs_dz = lapply(bs_Z, function(i){
  lapply(i, diff)
})
  
## latent and sensible heat flux profiles in parametrization
for(i in 1:length(fn)){
  for (j in 1:length(my.option)){
    # surface latent heat flux (W m-2)
    my.d[[i]][[j]]$LE.surf = my.d[[i]][[j]]$out.bound$lb_qw_flux[k] * Lsubl(t0[i]) * rho_sub
    # upper boundary: latent heat flux (W m-2)
    # LE.top  = out.bound$ub_qw_flux * Lsubl(BC_T_top[[1]]-273.15) * rho_sub
    # blowing snow latent heat exchange (W m-2) per layer and time
    my.d[[i]][[j]]$subl.drift = my.d[[i]][[j]]$out[k,,in.qv_src] * my.d[[i]][[j]]$bs_grid_height * my.d[[i]][[j]]$out[k,,in.lsub] * rho_sub
    # total per time 
    # LE.bs = apply(subl.drift, 2, sum)
    my.d[[i]][[j]]$LE.cum = cumsum(c(my.d[[i]][[j]]$LE.surf, my.d[[i]][[j]]$subl.drift))
    
    # surface sensible heat flux (W m-2)
    my.d[[i]][[j]]$H.surf = my.d[[i]][[j]]$out.bound$lb_Tw_flux[k] * Cp_air * rho_sub
    # upper boundary: sensible heat flux (W m-2)
    # H.top  = out.bound$ub_Tw_flux * Cp_air * rho_sub
    # blowing snow sensible heat exchange (W m-2) per layer and time
    my.d[[i]][[j]]$H.drift = my.d[[i]][[j]]$out[k,,in.t_src] * my.d[[i]][[j]]$bs_grid_height * Cp_air * rho_sub

    # total per time
    # H.bs = apply(H.drift, 2, sum)
    my.d[[i]][[j]]$H.cum = cumsum(c(my.d[[i]][[j]]$H.surf, my.d[[i]][[j]]$H.drift))
  }
}


# Load and post-process suspension set-ups or transport/default set-ups --------------------------------------------
my.d.susp = list()
for (i in 1:length(path.par.bs)){
  my.d.susp[[i]] = list()
  for (j in 1:length(path.par.bs[[1]])){
    if(exists("sub_meanR")) rm(sub_meanR)
    load(file = path.par.bs[[i]][j])
    my_meanR = if(exists("sub_meanR")){ sub_meanR }else{ meanR } # Some files store the mean radius in a variable called sub_meanR while others (with current code version) call it meanR
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[i]][j]) ){
      my.d.susp[[i]][[j]] = list(out=out, out.bound=out.bound, out.qi=out.qi, out.qni=out.qni, bs_grid_height=bs_grid_height, meanR=my_meanR,
                                 h_salt=h_salt, q_salt=q_salt, qn_salt=qn_salt, alpha_salt=alpha_salt, lambda_salt=lambda_salt)
    } else{
      my.d.susp[[i]][[j]] = list(out=out, out.bound=out.bound, out.qi=out.qi, out.qni=out.qni, bs_grid_height=bs_grid_height, meanR=my_meanR)
    }
  }
}
names(my.d.susp)     = case[1:length(path.par.bs)]

# fine levels
z.shift = 0.25
my.transform = function(x, offset){ log(offset + x) }
# heights of fine mesh
my.Z.fine = lapply(1:length(my.d.susp), function(i){ 
  lapply(1:length(my.d.susp[[i]]), function(j){
    if(grepl("log-spacing_with_15cm", path.par.bs[[i]][j])){
      Z_fine_with15cm
    } else{
      if(grepl("cm_high-res", path.par.bs[[i]][j])){
        my.hsalt = ifelse( 
          grepl("log-spacing_with_31cm_high-res", path.par.bs[[i]][j]), 0.2925, ifelse(
            grepl("log-spacing_with_14cm_high-res", path.par.bs[[i]][j]), 0.1275, ifelse(
              grepl("log-spacing_with_5cm_high-res", path.par.bs[[i]][j]), 0.0375, ifelse(
                grepl("log-spacing_with_8cm_high-res", path.par.bs[[i]][j]), 0.0675, ifelse(
                  grepl("log-spacing_with_16cm_high-res", path.par.bs[[i]][j]), 0.1425, stop("my.hsalt is not known."))
              ) ) ) )
        dz_bot_log = my.transform(0.015, z.shift) - my.transform(0, z.shift)
        exp( seq( my.transform(0, z.shift) + dz_bot_log, 0.95*my.transform(2.25 - my.hsalt, z.shift), dz_bot_log ) ) - z.shift + my.hsalt
      } else{
        Z_fine_linear
      }
    }
  })
})
# height of saltation layer (m)
h_salt_susp = lapply(1:length(my.d.susp), function(i){ 
  sapply(1:length(my.d.susp[[i]]), function(j){
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[i]][j]) ){
      my.d.susp[[i]][[j]]$h_salt
    } else{
      my.in = which.max( abs( my.d.susp[[i]][[j]]$bs_grid_height - 0.015 ) > 0.0001)
      if(grepl("cm_high-res", path.par.bs[[i]][j])) my.in = my.in - 1
      0.015 * my.in - 0.0075
    }
  })
})
# all heights (m)
bs_Z_susp = lapply(1:length(h_salt_susp), function(i){ 
  lapply(1:length(h_salt_susp[[i]]), function(j){
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[i]][j]) ){
      my.Z.salt = NULL
    } else{
      my.Z.salt = seq(from = 0.0075, by = 0.015, length.out = ceiling(h_salt_susp[[i]][j] / 0.015))
    }
    c( my.Z.salt, my.Z.fine[[i]][[j]], Z_coarse )
  })
})
# height differences
bs_dz_susp = lapply(bs_Z_susp, function(i){ lapply(i, diff) })
# UBC of bs mixing ratios
q_bs_top = lapply(my.run1, function(i){ 
  approx(x = i$ptcl$z, y = i$ptcl$mass.conc / rho_sub, xout = 11.25)$y
})
qn_bs_top = lapply(my.run1, function(i){ 
  # fit power law profile to heights from 2 to 5 m and extrapolate
  in.z.range = which(i$ptcl$z < 5 & i$ptcl$z > 2)
  my.var = i$ptcl$number.conc[in.z.range] / rho_sub
  my.x = log(i$ptcl$z[in.z.range])
  my.mod = lm( log(my.var) ~ my.x )
  var.pred = predict( my.mod, newdata = list(my.x = log(11.25)) )
  qn.bs.top = exp(var.pred)
})
# UBC of mean particle radius
meanR_ub = lapply(1:length(my.d.susp), function(i){
  sapply(1:length(my.d.susp[[i]]), function(j){
    if(parametrize.alpha[j]){
      alpha_ub = 3
    } else{
      alpha_ub = approx(x = my.run1[[i]]$ptcl$z, y = my.run1[[i]]$ptcl$gamma.shape, xout = 11.25 )$y
    }
    lambda_ub = ( pi*918.0/6.0 * qn_bs_top[[i]]/q_bs_top[[i]] * gamma(alpha_ub+3)/gamma(alpha_ub) )**(1.0/3.0)
    0.5 * alpha_ub / lambda_ub
  })
})
## standard deviation of particle diameter
sigma.dp.ub = list()
for(i in 1:length(my.d.susp)){
  sigma.dp.ub[[i]] = rep(NA, length(my.d.susp[[i]]))
  for (j in 1:length(my.d.susp[[i]])){
    if(parametrize.alpha[j]){
      alpha = 3
    } else{
      alpha = approx(x = my.run1[[i]]$ptcl$z, y = my.run1[[i]]$ptcl$gamma.shape, xout = c(bs_Z_susp[[i]][[j]], 11.25), rule = 2 )$y
    }
    my_q  = c(my.d.susp[[i]][[j]]$out.qi[k,], q_bs_top[[i]])
    my_qn = c(my.d.susp[[i]][[j]]$out.qni[k,], qn_bs_top[[i]])
    lambda = ( pi*918.0/6.0 * my_qn/my_q * gamma(alpha+3)/gamma(alpha) )**(1.0/3.0)
    # standard deviation of particle diameter (m)
    sigma.dp = sqrt( alpha / lambda^2 )
    my.d.susp[[i]][[j]]$sigma.dp = sigma.dp[1:(length(sigma.dp)-1)]
    sigma.dp.ub[[i]][j]          = sigma.dp[length(sigma.dp)]
  }
}
## latent heat flux profiles in suspension/transport/default set-ups
for(i in 1:length(my.d.susp)){
  for (j in 1:length(my.d.susp[[i]])){
    # surface latent heat flux (W m-2)
    my.d.susp[[i]][[j]]$LE.surf = my.d.susp[[i]][[j]]$out.bound$lb_qw_flux[k] * Lsubl(t0[i]) * rho_sub
    # blowing snow latent heat exchange (W m-2) per layer and time
    my.d.susp[[i]][[j]]$subl.drift = my.d.susp[[i]][[j]]$out[k,,in.qv_src] * my.d.susp[[i]][[j]]$bs_grid_height * my.d.susp[[i]][[j]]$out[k,,in.lsub] * rho_sub
    my.d.susp[[i]][[j]]$LE.cum = cumsum(c(my.d.susp[[i]][[j]]$LE.surf, my.d.susp[[i]][[j]]$subl.drift))
  }
}


# post-process LES data (reference) -----------------------------------------------

z.w = lapply(my.run1, function(i){ i$z.w })
dz.LES = lapply(z.w, diff)

## latent heat balance in LES
# LES: blowing snow latent heat exchange (W m-2) per layer
subl.drift.LES = lapply(1:length(case), function(i){
  my.run1[[i]]$ptcl$FQ_p / 1000 * dz.LES[[i]] * Lsubl(my.run1[[i]]$ta1$t) * rho_sub
})
# latent heat source (W m-3) from drifting/blowing snow (close to surface, use same layering as in parametrization)
S.lat.plot.LES       = lapply(1:length(case), function(i){
  subl.drift.LES[[i]] / dz.LES[[i]]
})
z.plot.LES           = lapply(1:length(case), function(i){
  my.run1[[i]]$ta1$z
})
# cumulative latent heat flux (W m-2)
LE.cum.plot.LES  = lapply(1:length(case), function(i){
  cumsum(c(my.run1[[i]]$boundary.flux$ql_surf, S.lat.plot.LES[[i]] * dz.LES[[i]]))
})

# LES: blowing snow sensible heat exchange (W m-2) per layer 
H.drift.LES = lapply(1:length(case), function(i){
  my.run1[[i]]$ptcl$Ft_p * dz.LES[[i]] * Cp_air * rho_sub
})
# sensible heat source (W m-3) from drifting/blowing snow (close to surface, use same layering as in parametrization)
S.sens.plot.LES  = lapply(1:length(case), function(i){
  H.drift.LES[[i]] / dz.LES[[i]]
})
# cumulative sensible heat flux (W m-2)
H.cum.plot.LES  = lapply(1:length(case), function(i){
  cumsum(c(my.run1[[i]]$boundary.flux$qh_surf, S.sens.plot.LES[[i]] * dz.LES[[i]]))
})


# Imitate MOST bulk approach using simulated T, q, and u at z = 1 m and z = 0 m -------------------------------

u.1m =    lapply(my.run1, function(i){ approx(x = i$ta1$z, y = i$ta1$u, xout = 1)$y }) # m s^-1
temp.1m = lapply(my.run1, function(i){ approx(x = i$ta1$z, y = i$ta1$t, xout = 1)$y }) # deg C
q.1m =    lapply(my.run1, function(i){ approx(x = i$ta1$z, y = i$ta1$q, xout = 1)$y / 1000 }) # g kg^-1


# loop with simulations
for(sim.number in (1:length(my.run1))){
  # iterative flux calculation
  MO.tmp = calc_fluxes_iter(z_ref_vw = 1, z_ref_scalar = 1, rough_len_m = rough_len_m_LES[sim.number], rough_len_t = rough_len_t_LES[sim.number], rough_len_q = rough_len_q_LES[sim.number],
                            vw_ref = u.1m[[sim.number]], T_ref = temp.1m[[sim.number]]+273.15, qv_ref = q.1m[[sim.number]], T_surf = t0[sim.number]+273.15, qv_surf = q0[sim.number] / 1000, 
                            FUN_psi_stable = calc_psi_stable_stearns_weidner, FUN_psi_unstable = calc_psi_unstable)
  qh.tmp = rho_sub * Cp_air * MO.tmp$Tw_flux
  ql.tmp = Lsubl(temp = t0[sim.number]) * rho_sub * MO.tmp$qw_flux
  # print and store results
  my.run1[[sim.number]]$imitate.MO.meas    = MO.tmp
  my.run1[[sim.number]]$imitate.MO.meas$qh = qh.tmp
  my.run1[[sim.number]]$imitate.MO.meas$ql = ql.tmp
  print(my.run1[[sim.number]]$imitate.MO.meas)
}
# relative error of bulk LE
sapply(1:4, function(i){ abs( ( my.run1[[i]]$imitate.MO.meas$ql - tail(LE.cum.plot.LES[[i]], n=1) ) / tail(LE.cum.plot.LES[[i]], n=1) ) })
# relative error of bulk H
sapply(1:4, function(i){ abs( ( my.run1[[i]]$imitate.MO.meas$qh - tail(H.cum.plot.LES[[i]], n=1) ) / tail(H.cum.plot.LES[[i]], n=1) ) })



# plot LE and H profiles --------------------------------------------------------------------
# Figure 5

# png(filename = "LE_parametriz_vs_LES_for_BLT.png", width = 7, height = 7, pointsize = 16, units = "in", res = 400)
pdf(file = "LE_H_parametriz_vs_LES_dx10cm.pdf", width = 9, height = 6.5)
par(mfrow = c(2,4), mar = c(4.5,1,0,0), oma = c(0,3,5.2,0.5), cex.axis = 1.2, cex.lab = 1.2)
ylims = c(0, 0.34)
for(plot_var in c("LE","H")){
  for(i in 1:length(case)){
    yticks = ifelse(i == 1, T, F) #i %% 2 
    if(plot_var == "LE"){
      xlims = range(c(0, sapply(1:length(my.option), function(k){ range(my.d[[i]][[k]]$LE.cum) }), LE.cum.plot.LES[[i]]))
      # xlims = range(c(0, sapply(2:length(my.option), function(k){ range(my.d[[i]][[k]]$LE.cum) }), LE.cum.plot.LES[[i]]))
      my_x  = LE.cum.plot.LES[[i]]
      my_xlab = expression(LE~(W~m^{-2}))
    } else{ # plot H
      xlims = range(c(0, sapply(1:length(my.option), function(k){ range(my.d[[i]][[k]]$H.cum) }), H.cum.plot.LES[[i]]))
      my_x  = H.cum.plot.LES[[i]] 
      my_xlab = expression(H~(W~m^{-2}))
    }
    plot(my_x, z.w[[i]], type = "o", col = 1, pch = 4, ylim = ylims, xlim = xlims,
         xlab = my_xlab, ylab = "", yaxt = "n")#, lwd = 2)
    axis(2, labels = yticks, )
    if(i == 1){
      mtext("z (m)", side = 2, line = 2.5, cex = 0.8)
    }
    # if(i > 2){
    # mtext(, side = 1, line = 2.5)
    # }
    # in.add        = which.min( abs( z_full_plot[1+sub_n_layers[[i]]] - z.w[[i]] ) )
    # in.marker.LES = c(1:sub_n_layers[[i]], in.add)
    # points(my_x[in.marker.LES], z.w[[i]][in.marker.LES], pch = 4, col = 1)
    inset.letter = if(i==4){ 0.1 }else{ 0.04 }
    # add label of subplot
    offset.letter = ifelse(plot_var=="LE", 0, 4)
    text(x = xlims[1] + inset.letter*(xlims[2]-xlims[1]), ylims[2] - 0.03*(ylims[2]-ylims[1]), labels = letters[i+offset.letter], cex = 1.5, font = 2)
    # parametrized latent or sensible heat flux
    for(j in 1:(length(my.option))){
      # for(j in 2:3){#1:(length(my.option))){
      if(plot_var == "LE"){
        my_x_par = my.d[[i]][[j]]$LE.cum
      } else{
        my_x_par = my.d[[i]][[j]]$H.cum
      }
      my_z = c(0, bs_Z[[i]][[j]][1:length(bs_dz[[i]][[j]])] + 0.5*bs_dz[[i]][[j]], 9)
      if( grepl(pattern = "no-salt-level", x = path.par[[i]][j]) ){
        my_z     = c( my_z[1], mean( c( my_h_salt[[i]], Z_fine_linear[1] ) ), my_z[-1] )
        my_x_par = c( rep(my_x_par[1], 2), my_x_par[-1] )
      }
      lines(my_x_par, my_z, type = "o", col = my.cols[j], pch = my.pch[j])
    }
    abline(v=0, col="darkgrey", lty=2)
    leg.pos = "topleft"#ifelse(plot_LE, "topleft", "topright")
    if( i == 1 & plot_var == "LE" ){ #(plot_LE & i == 2) | (!plot_LE & i == 1) ){
      legend(leg.pos, inset = c(0, -0.29), legend = expression( "LES-LM", "Existing:"~italic(u["*0"])*H*"-"*italic(Res)*L*"-"*italic(T)*Diag*"-"*italic(T[p])*Statio, 
                                                                italic(u["*0"])*L*"-"*italic(Res)*M, italic(u["*0"])*L*"-"*italic(Res)*M*"-"*italic(T)*Prog, 
                                                                "All_improved", "1_salt_level", "Zero line"), #italic(u["*0"])*L*"-"*italic(Res)*M*"-"*italic(T)*Prog*"-"*italic(T[p])*Trans, italic(u["*0"])*L*"-"*italic(Res)*M(1)*"-"*italic(T)*Prog*"-"*italic(T[p])*Trans
             lty = c(rep(1,6),2), pch = c(4,my.pch,NA), col = c(1,my.cols,"darkgrey"), bg="white", cex = 1.2, x.intersp = 0.5, xpd = NA, ncol = 3)
      # legend(leg.pos, inset = c(0, 0.13), legend = expression( "LES-LSM", "Existing", u["*0"]*A, u["*0"]*AB, u["*0"]*ABC, u["*0"]*tilde(A)*BC, "Zero line" ),
      #        lty = c(rep(1,6),2), pch = c(4,my.pch,NA), col = c(1,my.cols,"darkgrey"), bg="white", cex = 1.2, x.intersp = 0.5)
    }
  }
}
dev.off()

# write data to csv files
if(write2csv){
  df.out = list()
  for(i in 1:length(case)){
    z.exist    = c(0, bs_Z[[i]][[1]][1:length(bs_dz[[i]][[1]])] + 0.5*bs_dz[[i]][[1]], 9)
    z.full.ABC = c(0, bs_Z[[i]][[4]][1:length(bs_dz[[i]][[4]])] + 0.5*bs_dz[[i]][[4]], 9)
    z.tilde    = c(0, bs_Z[[i]][[5]][1:length(bs_dz[[i]][[5]])] + 0.5*bs_dz[[i]][[5]], 9)
    na.exist = rep(NA, length(z.full.ABC) - length(z.exist) )
    na.tilde = rep(NA, length(z.full.ABC) - length(z.tilde) )
    option.lab = c("ustar0L_ResM","ustar0L_ResM_TProg","all_improved")
    df.out[[i]] = data.frame(Case = rep(case.label[i], length(z.full.ABC)),
                        z_stag = z.full.ABC,
                        z_stag_existing = c(z.exist, na.exist),
                        z_stag_1_salt_level = c(z.tilde, na.tilde),
                        LE_existing = c(my.d[[i]][[1]]$LE.cum, na.exist))
    LE_tmp = as.data.frame( sapply(2:4, function(j){ my.d[[i]][[j]]$LE.cum }) )
    names(LE_tmp) = paste("LE_", option.lab, sep="")
    LE_1_salt_level = c(my.d[[i]][[5]]$LE.cum, na.tilde)
    H_existing = c(my.d[[i]][[1]]$H.cum, na.exist)
    H_tmp = as.data.frame( sapply(2:4, function(j){ my.d[[i]][[j]]$H.cum }) )
    names(H_tmp) = paste("H_", option.lab, sep="")
    H_1_salt_level = c(my.d[[i]][[5]]$H.cum, na.tilde)
    df.out[[i]] = cbind(df.out[[i]], LE_tmp, LE_1_salt_level, 
                   H_existing, H_tmp, H_1_salt_level)
    # round values digits = 4, digits = 2
    df.out[[i]][,2:4] = round(df.out[[i]][,2:4], digits = 4)
    df.out[[i]][,5:NCOL(df.out[[i]])] = round(df.out[[i]][,5:NCOL(df.out[[i]])], digits = 2)
  }
  df.out = do.call(rbind, df.out)
  write.csv(df.out, file = paste("Fig_5_1Dmodel.csv",sep=""), na = "NaN", row.names = F)
}

# mean errors for each 1D model set-up ---------------------------------------
# Figure 7

# mean absolute error of LE (W m-2)
mae.LE = sapply(1:length(my.option), function(j){
  mean( sapply(1:length(case), function(i){
    abs( tail(my.d[[i]][[j]]$LE.cum, n=1) - LE.cum.plot.LES[[i]][which.min(abs(z.w[[i]]-9))] )
  }) )
})

# mean absolute error of H (W m-2)
mae.H = sapply(1:length(my.option), function(j){
  mean( sapply(1:length(case), function(i){
    abs( tail(my.d[[i]][[j]]$H.cum, n=1) - H.cum.plot.LES[[i]][which.min(abs(z.w[[i]]-9))] )
  }) )
})

lab.setup = expression( "Existing", 
                        italic(u["*0"])*L*"-"*italic(Res)*M, 
                        italic(u["*0"])*L*"-"*italic(Res)*M*"-"*italic(T)*Prog, #atop(italic(u["*0"])*L*"-"*italic(Res)*M*"-", italic(T)*Prog), 
                        "All_improved", #atop(italic(u["*0"])*L*"-"*italic(Res)*M*"-", italic(T)*Prog*"-"*italic(T[p])*Trans), 
                        "1_salt_level" ) #atop(italic(u["*0"])*L*"-"*italic(Res)*M(1)*"-", italic(T)*Prog*"-"*italic(T[p])*Trans) )
my.dx = 3
y.max = max(c(mae.LE, mae.H))
pdf(file = "MAE_1D_model_LE_H_dx10cm.pdf", width = 8, height = 3.2)
par(mar=c(2,4.3,0.5,1.5))
barplot(rbind(mae.LE, mae.H), width = 0.5, beside = T, space = c(0,my.dx), col = c(1,"grey"), xlab = "", 
        ylab = expression(MAE~(W~m^{-2})), ylim = c(0-0.02*y.max, 1.02*y.max), axes = F,
        names.arg = lab.setup, #NULL, 
        legend.text = expression(LE, H), args.legend = list(title="MAE of"), xpd = NA)
axis(side = 2, at = seq(0, ceiling(y.max), by = 2))
axis(side = 4, at = seq(0, ceiling(y.max), by = 2), labels = F)
# text(seq(2,by=my.dx-0.5,length.out=length(lab.setup)), -3, labels = lab.setup, adj = c(0.5, 0.5), xpd = NA)
box()
dev.off()

if(write2csv){
  setup  = c("Existing", "ustar0L_ResM","ustar0L_ResM_TProg","All_improved", "1_salt_level")
  df.out = round( as.data.frame(cbind(mae.LE, mae.H)), digits = 2 )
  df.out = cbind(setup, df.out)
  colnames(df.out) = c("Set-up", "MAE_LE", "MAE_H")
  write.csv(df.out, file = paste("Fig_7_MAE_1Dmodel.csv",sep=""), na = "NaN", row.names = F)
}

# plot T, q, RH profiles --------------------------------------------------
# Figures 6 or S7
in.case.plot = 1
# in.case.plot = 2:4

RH.ax.at = list(seq(86, 100, 2),
                seq(98.5, 100, 0.25),
                seq(99.825, 100, 0.025),
                seq(99.65, 100, 0.05))

if(identical(in.case.plot, 1)){
  pdf(file = "T_q_RH_parametriz_vs_LES_dx10cm_case1.pdf", width = 7.4, height = 2.7)
  par(mfrow = c(1,3), mar = c(4,1,5,0), oma = c(0,3,0,1), cex.axis = 1.1, cex.lab = 1.1)
  leg.inset.y = -0.42
}
if(identical(in.case.plot, 2:4)){
  pdf(file = "T_q_RH_parametriz_vs_LES_dx10cm_case2-3b.pdf", width = 7.4, height = 7)
  par(mfrow = c(3,3), mar = c(1.5,1,1,0), oma = c(2.5,3,4,1), cex.axis = 1.1, cex.lab = 1.1)
  leg.inset.y = -0.37
}
ylims = c(0, 2.5)# c(0, 9)
in.options = c(2,1,3:5) #1:length(my.option)
for(i in in.case.plot){
  # plot temperature
  my.x = c(t0[i], my.run1[[i]]$ta1$t)
  my.y = c(0, z.plot.LES[[i]])
  xlims = range( c(t0[i], sapply(1:length(my.option), function(j){ #t9m[i], 
    in.z = which(bs_Z[[i]][[j]] < ylims[2])
    my.d[[i]][[j]]$out[k, in.z, in.t] - 273.15 
  })) )
  plot(my.x, my.y, type = "l", lwd = 2, ylim = ylims, xlim = xlims, xlab = "", ylab = "")
  title(ylab = expression(italic(z)~(m)), xpd = NA, line = 2.5)
  # lines(my.x, my.y)
  t.pnt = approx(x = my.y, y = my.x, xout = c(0, bs_Z[[i]][[3]], 9))$y
  points(t.pnt, c(0, bs_Z[[i]][[3]], 9), pch = 4, cex = 2, lwd = 2)
  # axis(3, labels = F)
  # abline(h=my_h_salt[i],lty=2, col="darkgrey")
  abline(h=2.25,lty=3)
  for(j in in.options){
    z.plot = c(0, bs_Z[[i]][[j]], 9)
    lines(c(t0[i], my.d[[i]][[j]]$out[k, , in.t] - 273.15, t9m[i]), z.plot, type = "o", col = my.cols[j], pch = my.pch[j])
  }
  txt.inset = 0.75 #ifelse(i==1, 0.75, 0.25)
  text(x = xlims[1] + txt.inset*diff(xlims), ylims[2] - 0.03*diff(ylims), labels = letters[seq(1,10,3)][i-in.case.plot[1]+1], cex = 1.5, font = 2)
  if( i == in.case.plot[1] ){
    legend("topleft", inset = c(-0.23, leg.inset.y), legend = expression( "LES-LM", "Existing:"~italic(u["*0"])*H*"-"*italic(Res)*L*"-"*italic(T)*Diag*"-"*italic(T[p])*Statio, 
                                                               italic(u["*0"])*L*"-"*italic(Res)*M, italic(u["*0"])*L*"-"*italic(Res)*M*"-"*italic(T)*Prog, 
                                                               "All_improved", #italic(u["*0"])*L*"-"*italic(Res)*M*"-"*italic(T)*Prog*"-"*italic(T[p])*Trans, 
                                                               "1_salt_level", #italic(u["*0"])*L*"-"*italic(Res)*M(1)*"-"*italic(T)*Prog*"-"*italic(T[p])*Trans, 
                                                               1*st~coarse~level),
           lty = c(rep(1,6),3), pch = c(4,my.pch,NA), col = c(1,my.cols,1), lwd = c(2,rep(1,6)), bg="white", cex = 1.1, x.intersp = 0.5, ncol = 3, xpd = NA)
  }
  if(i == tail(in.case.plot,n=1)) title(xlab = expression(italic("T")~(degree*C)), xpd = NA)
  # specific humidity
  my.x = c(q0[i], my.run1[[i]]$ta1$q)
  xlims = range( c(q0[i], sapply(1:length(my.option), function(j){ #q9m[i], 
    in.z = which(bs_Z[[i]][[j]] < ylims[2])
    my.d[[i]][[j]]$out[k, in.z, in.qv] * 1000 
  })) )
  plot(my.x, my.y, type = "l", lwd = 2, ylim = ylims, xlim = xlims, xlab = "", ylab = "", yaxt = "n")
  # lines(my.x, my.y)
  x.pnt = approx(x = my.y, y = my.x, xout = c(0, bs_Z[[i]][[3]], 9))$y
  points(x.pnt, c(0, bs_Z[[i]][[3]], 9), pch = 4, cex = 2, lwd = 2)
  axis(2, labels = F)
  # axis(3, labels = F)
  abline(h=2.25,lty=3)
  for(j in in.options){
    z.plot = c(0, bs_Z[[i]][[j]], 9)
    lines(c(q0[i], my.d[[i]][[j]]$out[k, , in.qv] * 1000, q9m[i]), z.plot, type = "o", col = my.cols[j], pch = my.pch[j])
  }
  text(x = xlims[1] + txt.inset*diff(xlims), ylims[2] - 0.03*diff(ylims), labels = letters[seq(2,11,3)][i-in.case.plot[1]+1], cex = 1.5, font = 2)
  if(i == tail(in.case.plot,n=1)) title(xlab = expression(italic(q)~(g~kg^{-1})), xpd = NA)
  # RH profiles
  my.x = c(RH0[i], my.run1[[i]]$ta1$rh)
  xlims = range( c(RH0[i], sapply(1:length(my.option), function(j){ #RH9m[i], 
    in.z = which(bs_Z[[i]][[j]] < ylims[2])
    my.d[[i]][[j]]$out[k, in.z, in.RH] 
  })) )
  plot(my.x, my.y, type = "l", lwd = 2, ylim = ylims, xlim = xlims, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
  # lines(my.x, my.y)
  x.pnt = approx(x = my.y, y = my.x, xout = c(0, bs_Z[[i]][[3]], 9))$y
  points(x.pnt, c(0, bs_Z[[i]][[3]], 9), pch = 4, cex = 2, lwd = 2)
  in.ax.lab = if(i==1){ 
    1:length(RH.ax.at[[i]]) 
  } else{
    if(i==2){
      seq(1,length(RH.ax.at[[i]]),2)
    } else{
      seq(2,length(RH.ax.at[[i]]),2) 
    }
  }
  RH.ax.lab = rep(NA, length(RH.ax.at[[i]]))
  RH.ax.lab[in.ax.lab] = RH.ax.at[[i]][in.ax.lab]
  axis(1, at = RH.ax.at[[i]], labels = RH.ax.lab)#, xpd = NA)
  axis(2, labels = F)
  # axis(3, labels = F)
  abline(h=2.25,lty=3)
  for(j in in.options){
    z.plot = c(0, bs_Z[[i]][[j]], 9)
    lines(c(RH0[i], my.d[[i]][[j]]$out[k, , in.RH], RH9m[i]), z.plot, type = "o", col = my.cols[j], pch = my.pch[j])
  }
  text(x = xlims[1] + txt.inset*diff(xlims), ylims[2] - 0.03*diff(ylims), labels = letters[seq(3,12,3)][i-in.case.plot[1]+1], cex = 1.5, font = 2)
  if(i == tail(in.case.plot,n=1)) title(xlab = expression(italic(RH)~("%")), xpd = NA)
}
dev.off()

# write data to csv files
if(write2csv){
  df.out = list()
  for(i in 1:length(case)){
    z.half.exist = c(0, bs_Z[[i]][[1]], 9)
    z.half.ABC   = c(0, bs_Z[[i]][[4]], 9)
    z.half.tilde = c(0, bs_Z[[i]][[5]], 9)
    na.exist = rep(NA, length(z.half.ABC) - length(z.half.exist) )
    na.tilde = rep(NA, length(z.half.ABC) - length(z.half.tilde) )
    option.lab = c("ustar0L_ResM","ustar0L_ResM_TProg","all_improved")
    df.out[[i]] = data.frame( Case = rep(case.label[i], length(z.half.ABC)),
                        z          = z.half.ABC,
                        z_existing = c(z.half.exist, na.exist),
                        z_1_salt_level    = c(z.half.tilde, na.tilde),
                        t_existing = c(t0[i], my.d[[i]][[1]]$out[k, , in.t] - 273.15, t9m[i], na.exist) )
    t_tmp = as.data.frame( sapply(2:4, function(j){ c(t0[i], my.d[[i]][[j]]$out[k, , in.t] - 273.15, t9m[i]) }) )
    names(t_tmp) = paste("t_", option.lab, sep="")
    t_1_salt_level = c(t0[i], my.d[[i]][[5]]$out[k, , in.t] - 273.15, t9m[i], na.tilde)
    q_existing = c(q0[i], 1000 * my.d[[i]][[1]]$out[k, , in.qv], q9m[i], na.exist)
    q_tmp = as.data.frame( sapply(2:4, function(j){ c(q0[i], 1000 * my.d[[i]][[j]]$out[k, , in.qv], q9m[i]) }) )
    names(q_tmp) = paste("q_", option.lab, sep="")
    q_1_salt_level = c(q0[i], 1000 * my.d[[i]][[5]]$out[k, , in.qv], q9m[i], na.tilde)
    RH_existing = c(RH0[i], my.d[[i]][[1]]$out[k, , in.RH], RH9m[i], na.exist)
    RH_tmp = as.data.frame( sapply(2:4, function(j){ c(RH0[i], my.d[[i]][[j]]$out[k, , in.RH], RH9m[i]) }) )
    names(RH_tmp) = paste("RH_", option.lab, sep="")
    RH_1_salt_level = c(RH0[i], my.d[[i]][[5]]$out[k, , in.RH], RH9m[i], na.tilde)
    df.out[[i]] = cbind(df.out[[i]], t_tmp, t_1_salt_level, 
                   q_existing, q_tmp, q_1_salt_level,
                   RH_existing, RH_tmp, RH_1_salt_level)
    # round values
    df.out[[i]][, -1] = round(df.out[[i]][, -1], digits = 4)
  }
  df.out = do.call(rbind, df.out)
  write.csv(df.out, file = paste("Fig_6_S7_1Dmodel.csv",sep=""), na = "NaN", row.names = F)
}



# Plot ratio between latent and sensible heat exchange in LES -------------
# Figure S3

pdf(file = "bs_heat_ratio_LES_dx10cm.pdf", width = 8, height = 5.2)
par(mfrow = c(1,2), mar = c(5,1.5,1,0), oma = c(0,2.5,0,1), cex.axis = 1.1)
my.i = 1:2
my.j = c(4,3)
my.ratio = lapply(1:length(S.lat.plot.LES), function(x){ 
  if(x %in% c(my.i[1],my.j[1])){
    -S.lat.plot.LES[[x]] / S.sens.plot.LES[[x]]
  } else{
    -S.sens.plot.LES[[x]] / S.lat.plot.LES[[x]]
  }
})
xlabs = expression(-frac(italic(L[s]~S[v]),italic(c[p]~S["T"])), -frac(italic(c[p]~S["T"]),italic(L[s]~S[v])))
ytick.lab = c(T,F)
col.case = c(1,2,4,"orange")
lab.case = c("Case 1","Case 2","Case 3a","Case 3b")
for(i in 1:length(my.i)){
  my.in = c(my.i[i], my.j[i])
  xlims = range(sapply(my.ratio[my.in], range))
  plot(my.ratio[[my.in[1]]], my.run1[[my.in[1]]]$ptcl$z, type = "l", log = "y", xlim = xlims, xlab = "", ylab = "", 
       yaxt = "n", col = col.case[my.in[1]], xpd = NA)
  axis(2, labels = ytick.lab[i])
  if(i == 1) title(ylab = expression(italic(z)~(m)), xpd = NA)
  title(xlab = xlabs[i], line = 4)
  lines(my.ratio[[my.in[2]]], my.run1[[my.in[2]]]$ptcl$z, col = col.case[my.in[2]])
  abline(v=1, lty=2, col="darkgrey")
  abline(h=0.3, lty = 3)#, col="darkgrey")
  axis(side = 4, labels = F)
  legend("topleft", legend = c(as.expression(bquote(.(lab.case[my.in]))), "Thermal equilibrium", expression(italic(z)~"="~0.3~m)),
         inset = c(0,0.1), col = c(col.case[my.in], "darkgrey", 1), lty = c(1,1,2,3), cex = 1.1)
  text(x = xlims[1] + 0.04*(xlims[2]-xlims[1]), 17, labels = letters[i], cex = 1.1, font = 2)
}
dev.off()


# Plot vertical LES profiles for comparison with measurements -------------
# Figure 2

# convert units of mass flux from kg m-2 min-1 to g m-2 s-1
mass_hflux_les = lapply(my.run1, function(i){ 1000/60*i$ptcl$mass_hflux })
flux_spc       = lapply(my.meas, function(i){ 1000/60*i$spc.flux })
color = c(1,2,4,"orange")
pchs  = c(16,8,23,4)

pdf(file = file.path(path.LES, "profiles_dx10cm_vs_meas_4cases.pdf"), width = 6.6, height = 6.1)
par(mfrow = c(2,4), mar = c(4.8,0.8,0.5,0), oma = c(0,3,5,1), cex.lab = 1.1) #cex.axis = 1.1, 

# wind speed: average vertical profile
ylims1 = c(0,9)
ylims2 = c(0,0.3)
in.z1 = which(my.run1[[1]]$ta1$z < ylims1[2])
xlims = range( c(0, sapply(my.run1, function(i){ range(i$ta1$u[in.z1]) })) )
plot(c(0,my.run1[[1]]$ta1$u), c(0,my.run1[[1]]$ta1$z), type = "n", xlim = xlims, ylim = ylims1, xlab = expression(italic(u)~(m~s^{-1})), ylab = "")
title(ylab=expression(italic(z)~(m)), line = 2.2, xpd = NA)
abline(v = 0, col = "darkgrey", lty = 4)
for(i in 1:length(my.run1)){
  lines(c(0,my.run1[[i]]$ta1$u), c(0,my.run1[[i]]$ta1$z), type = "l", col = color[i])
  col.bg = if(i == 3){ color[4] }else{ color[i] }
  if(i < 4){
    arrows(x0 = rep(my.meas[[i]]$u, each = 2), y0 = rep(my.meas[[i]]$z.u, each = 2), x1 = rep(my.meas[[i]]$u, each = 2) + rep(my.unc[[i]]$u, each = 2) * rep(c(1,-1),3), y1 = rep(my.meas[[i]]$z.u, each = 2), col = color[i], angle = 90, length = 0.05)
    points(my.meas[[i]]$u, my.meas[[i]]$z.u, col=color[i], pch = pchs[i], bg = color[4])
  }
}
axis(side = 3, labels = NA)
text(x = xlims[1] + 0.04*diff(xlims), ylims1[2] - 0.03*diff(ylims1), labels = "a", cex = 1.1, font = 2)
legend("topleft", inset = c(0,-0.33), ncol = 4, legend =  expression(Case~1~simulation, "Case 1 measurement", NA, Case~2~simulation, "Case 2 measurement", NA, Case~3*a~simulation, Case~3*b~simulation, "Case 3 measurement","MOST bulk flux","Zero line"), 
       lty = c(1,NA, NA, 1,NA, NA, 1,1,NA,3,4), pch = c(NA,pchs[1],NA, NA, pchs[2],NA, NA, NA,pchs[3],NA,NA), col = c(color[c(1,1, NA, 2,2, NA, 3,4,3)],1,"darkgrey"), lwd = c(1,1, NA, 1,1, NA, 1,1,1,1.3,1), pt.bg = color[4], xpd = NA) #cex = 0.9, y.intersp = 1
# temperature: average vertical profile
xlims = c(-6, -2.57) #range( c(t0, sapply(my.meas, function(i){ range(i$t) }), sapply(my.run1, function(i){ range(i$ta1$t[in.z1]) })) )
plot( t0[1], 0, type = "n", xlab = expression(italic("T")~(phantom()^{degree}*C)), xlim = xlims, ylim = ylims1, ylab = "", yaxt = "n")
axis(2, labels = F)
axis(side = 3, labels = NA)
for(i in 1:length(my.run1)){
  lines(c(t0[i],my.run1[[i]]$ta1$t), c(0,my.run1[[i]]$ta1$z), type = "l", col = color[i])
  if(i < 4){
    arrows(x0 = rep(my.meas[[i]]$t, each = 2), y0 = rep(my.meas[[i]]$z.t, each = 2), x1 = rep(my.meas[[i]]$t, each = 2) + rep(my.unc[[i]]$t, each = 2) * rep(c(1,-1),3), y1 = rep(my.meas[[i]]$z.t, each = 2), col = color[i], lty = rep(c(1,1,2),each=2), angle = 90, length = 0.05)
    points(my.meas[[i]]$t, my.meas[[i]]$z.t, col=color[i], pch = pchs[i], bg = color[4])
  }
}
text(x = xlims[1] + 0.04*diff(xlims), ylims1[2] - 0.03*diff(ylims1), labels = "b", cex = 1.1, font = 2)
# spec humidity: average vertical profile
xlims = c(2.51, 3.38) #range( sapply(my.meas, function(i){ range(i$q) }) )
plot( q0.meas[1], 0, type = "n", xlab = expression(italic(q)~(g~kg^{-1})), xlim = xlims, ylim = ylims1, ylab = "", yaxt = "n")
axis(2, labels = F)
axis(side = 3, labels = NA)
for(i in 1:length(my.run1)){
  lines(c(q0.meas[i],my.run1[[i]]$ta1$q), c(0,my.run1[[i]]$ta1$z), type = "l", col = color[i])
  if(i < 4){
    arrows(x0 = rep(my.meas[[i]]$q, each = 2), y0 = rep(my.meas[[i]]$z.q, each = 2), x1 = rep(my.meas[[i]]$q, each = 2) + rep(my.unc[[i]]$q, each = 2) * rep(c(1,-1),2), y1 = rep(my.meas[[i]]$z.q, each = 2), col = color[i], angle = 90, length = 0.05)
    points(my.meas[[i]]$q, my.meas[[i]]$z.q, col=color[i], pch = pchs[i], bg = color[4])
  }
}
text(x = xlims[1] + 0.04*diff(xlims), ylims1[2] - 0.03*diff(ylims1), labels = "c", cex = 1.1, font = 2)
# relative humidity RH
xlims = range( c( sapply(my.run1, function(i){ min(i$ta1$rh[in.z1]) }), sapply(my.meas, function(i){ range(i$rh) }), 100) )
plot( 100, 0, type = "n", xlab = expression(italic(RH)~("%")), ylab = "", xlim = xlims, ylim = ylims1, yaxt = "n")
axis(2, labels = F)
axis(side = 3, labels = NA)
for(i in 1:length(my.run1)){
  lines(c(100,my.run1[[i]]$ta1$rh), c(0,my.run1[[i]]$ta1$z), type = "l", col = color[i])
  if(i < 4){
    arrows(x0 = rep(my.meas[[i]]$rh, each = 2), y0 = rep(my.meas[[i]]$z.rh, each = 2), x1 = rep(my.meas[[i]]$rh, each = 2) + rep(my.unc[[i]]$rh, each = 2) * rep(c(1,-1),2), y1 = rep(my.meas[[i]]$z.rh, each = 2), col = color[i], angle = 90, length = 0.05)
    points(my.meas[[i]]$rh, my.meas[[i]]$z.rh, col=color[i], pch = pchs[i], bg = color[4])
  }
}
text(x = xlims[1] + 0.04*diff(xlims), ylims1[2] - 0.03*diff(ylims1), labels = "d", cex = 1.1, font = 2)
# horizontal mass flux: average vertical profile (with log axis)
xlims = range( sapply(mass_hflux_les, function(i){ range(i[1:20]) }) )
log.xaxtick = seq(floor(log10(xlims[1])), ceiling(log10(xlims[2])), 1)
xaxlab  = sapply(log.xaxtick, function(i){ as.expression(bquote(10^.(i))) })
xaxtick.small = do.call(c, lapply(10^log.xaxtick[-length(log.xaxtick)], function(i){i*(2:9)}))
plot(mass_hflux_les[[1]], my.run1[[1]]$ta1$z, ylim = ylims2, xlim = xlims, type = "n", log = "x", xlab = "", ylab = "", xaxt = "n")
title(ylab=expression(italic(z)~(m)), line = 2.2, xpd = NA)
title(xlab=expression(Flux~(g~m^{-2}~s^{-1})), line = 3, xpd = NA)
axis(side = 1, at = 10^log.xaxtick, labels = xaxlab)
axis(1, at= xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.2)
axis(side = 3, at = 10^log.xaxtick, labels = NA)
axis(3, at= xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.2)
for(i in 1:length(my.run1)){
  lines(mass_hflux_les[[i]], my.run1[[i]]$ta1$z, type = "l", col = color[i])#, pch = 20, cex = 0.7)
  in.spc = if(i==1){ 1:2 }else{ 2 }
  if(i < 4){
    arrows(x0 = flux_spc[[i]][in.spc], y0 = my.meas[[i]]$z.spc[in.spc] - 0.02, x1 = flux_spc[[i]][in.spc], y1 = my.meas[[i]]$z.spc[in.spc] + 0.02, col = color[i], angle = 90, length = 0.05, code = 3, lty = 2)
    points(flux_spc[[i]][in.spc], my.meas[[i]]$z.spc[in.spc], col=color[i], pch = pchs[i], bg = color[4], cex = ifelse(i==2, 1.5, 1))
  }
}
text(x = xlims[1] * (xlims[2]/xlims[1])^0.04, ylims2[2] - 0.03*diff(ylims2), labels = "e", cex = 1.1, font = 2)
# horizontal particle number flux: average vertical profile at given times
xlims = range( sapply(my.run1, function(i){ range(i$ptcl$number_hflux[1:20]) }) )
log.xaxtick = seq(floor(log10(xlims[1])), ceiling(log10(xlims[2])), 1)
xaxlab  = sapply(log.xaxtick, function(i){ as.expression(bquote(10^.(i))) })
xaxtick.small = do.call(c, lapply(10^log.xaxtick[-length(log.xaxtick)], function(i){i*(2:9)}))
plot(1, 1, ylim = ylims2, xlim = xlims, type = "n", log = "x", xlab = "", ylab = "", xaxt = "n", yaxt = "n")
axis(2, labels = F)
title(xlab=expression(italic(F[N])~(cm^{-2}~s^{-1})), line = 3, xpd = NA)
axis(side = 1, at = 10^log.xaxtick, labels = xaxlab)
axis(1, at= xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.2)
axis(side = 3, at = 10^log.xaxtick, labels = NA)
axis(3, at= xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.2)
for(i in 1:length(my.run1)){
  lines(my.run1[[i]]$ptcl$number_hflux, my.run1[[i]]$ta1$z, type = "l", col = color[i])#, pch = 20, cex = 0.7)
  in.spc = if(i==1){ 1:2 }else{ 2 }
  if(i < 4){
    arrows(x0 = my.meas[[i]]$spc.number.flux[in.spc], y0 = my.meas[[i]]$z.spc[in.spc] - 0.02, x1 = my.meas[[i]]$spc.number.flux[in.spc], y1 = my.meas[[i]]$z.spc[in.spc] + 0.02, col = color[i], angle = 90, length = 0.05, code = 3, lty = 2)
    points(my.meas[[i]]$spc.number.flux[in.spc], my.meas[[i]]$z.spc[in.spc], col=color[i], pch = pchs[i], bg = color[4], cex = ifelse(i==2, 1.5, 1))
  }
}
text(x = xlims[1] * (xlims[2]/xlims[1])^0.04, ylims2[2] - 0.03*diff(ylims2), labels = "f", cex = 1.1, font = 2)
# cumulative sensible heat exchange as a function of height
xlims = range( c(0, sapply(my.meas, function(i){ range(i$H$EC) }), sapply(H.cum.plot.LES, function(i){ range(i) })) )
plot(H.cum.plot.LES[[1]], z.w[[1]], type = "n", xlim = xlims, ylim = c(0,0.3), xlab = expression(italic(H)~(W~m^{-2})), ylab = "", yaxt = "n")
axis(2, labels = F)
axis(side = 3, labels = NA)
abline(v = 0, col = "darkgrey", lty = 4)
for(i in 1:length(my.run1)){
  lines(H.cum.plot.LES[[i]], z.w[[i]], type = "l", col = color[i])
  col.bg = if(i == 3){ color[4] }else{ color[i] }
  if(i < 4) points(x = my.meas[[i]]$H$EC, y = par("usr")[4]+0.0095, pch = pchs[i], bg = color[4], cex = 1.2, xpd = NA, col = color[i])
  lines(x = rep(my.run1[[i]]$imitate.MO.meas$qh, 2), y = c(0, 0.5), col = color[i], lty = 3, lwd = 1.3)
}
text(x = xlims[1] + 0.04*diff(xlims), ylims2[2] - 0.03*diff(ylims2), labels = "g", cex = 1.1, font = 2)
# cummulative latent heat exchange as a function of height
xlims = range( c(0, sapply(my.meas, function(i){ range(i$LE$EC) }), sapply(LE.cum.plot.LES, function(i){ range(i) })) )
plot(LE.cum.plot.LES[[1]], z.w[[1]], ylim = ylims2, xlim = xlims, type = "n", xlab = expression(italic(LE)~(W~m^{-2})), ylab = "", yaxt = "n")
axis(2, labels = F)
axis(side = 3, labels = NA)
abline(v = 0, col = "darkgrey", lty = 4)
for(i in 1:length(my.run1)){
  lines(LE.cum.plot.LES[[i]], z.w[[i]], type = "l", col = color[i])#, pch = 20, cex = 0.7)
  col.bg = if(i == 3){ color[4] }else{ color[i] }
  if(i < 4) points(x = my.meas[[i]]$LE$EC, y = par("usr")[4]+0.0095, pch = pchs[i], bg = color[4], cex = 1.2, xpd = NA, col = color[i])
  lines(x = rep(my.run1[[i]]$imitate.MO.meas$ql, 2), y = c(ifelse(i==3, 0.0045, 0), 0.5), col = color[i], lty = 3, lwd = 1.3)
}
text(x = xlims[1] + 0.04*diff(xlims), ylims2[2] - 0.03*diff(ylims2), labels = "h", cex = 1.1, font = 2)
dev.off()


if(write2csv){
  # measured data
  case.lab.meas = 1:3
  z_tmp = c(0, my.meas[[1]]$z.spc[2:1], my.meas[[1]]$z.u)
  df.out = data.frame( Case = rep(case.lab.meas, each = length(z_tmp)),
                       z    = rep(z_tmp, 3) )
  u_tmp = data.frame( u = do.call(c, lapply(1:3, function(i){ c( rep(NA, 3), my.meas[[i]]$u ) }) ) )
  df.out = cbind.data.frame(df.out, u_tmp)
  df.out$uncty_u = rep( c( rep(NA, 3), my.unc[[1]]$u ), 3 )
  t_tmp = data.frame( t = do.call(c, lapply(1:3, function(i){ c( my.meas[[i]]$t[1], NA,NA, my.meas[[i]]$t[2:3], NA ) }) ) )
  df.out = cbind.data.frame(df.out, t_tmp)
  df.out$uncty_t = rep( c( my.unc[[1]]$t[1], NA,NA, my.unc[[1]]$t[2:3], NA ), 3 )
  q_tmp = data.frame( q = do.call(c, lapply(1:3, function(i){ c( my.meas[[i]]$q[1], NA,NA, my.meas[[i]]$q[2], NA,NA ) }) ) )
  uncty_q = data.frame( uncty_q = do.call(c, lapply(1:3, function(i){ c( my.unc[[i]]$q[1], NA,NA, my.unc[[i]]$q[2], NA,NA ) }) ) )
  RH_tmp = data.frame( RH = do.call(c, lapply(1:3, function(i){ c( rep(NA, 3), my.meas[[i]]$rh, NA,NA ) }) ) )
  uncty_RH = data.frame( uncty_RH = do.call(c, lapply(1:3, function(i){ c( rep(NA, 3), my.unc[[i]]$rh, NA,NA ) }) ) )
  sn_flx_case1 = c( NA, round(flux_spc[[1]][2:1], digits = 1), rep(NA, 3) )
  sn_flx = data.frame( sn_flx = c( sn_flx_case1, do.call(c, lapply(2:3, function(i){ c( NA, round(flux_spc[[i]][2], digits = 1), rep(NA, 4) ) }) ) ) )
  df.out = cbind.data.frame(df.out, q_tmp, uncty_q, RH_tmp, uncty_RH, sn_flx)
  df.out$uncty_z_sn_flx = c( NA, 0.02, 0.02, rep(NA,3), rep( c(NA, 0.02, rep(NA,4)), 2 ) )
  H_tmp = data.frame( H = do.call(c, lapply(1:3, function(i){ c( rep(NA, 4), my.meas[[i]]$H$EC, NA ) }) ) )
  LE_tmp = data.frame( LE = do.call(c, lapply(1:3, function(i){ c( rep(NA, 4), my.meas[[i]]$LE$EC, NA ) }) ) )
  df.out = cbind.data.frame(df.out, H_tmp, LE_tmp)
  write.csv(df.out, file = "Fig_2_measurements.csv", na = "NaN", row.names = F)
}

# Particle properties in all four LES simulations (vertical profiles) -----------------------
# Figure 4

ylims = c(0.0075,11.25)
color = c(1,2,4,"orange")
log.q.plot = "xy"
log.dp.plot= "y"
log.q.plot = paste(log.q.plot,"y",sep="")
log.dp.plot = paste(log.dp.plot,"y",sep="")
log.yaxtick = seq(floor(log10(ylims[1])), ceiling(log10(ylims[2])), 1)
yaxlab  = sapply(log.yaxtick, function(i){ as.expression(bquote(10^.(i))) })
yaxtick.small = do.call(c, lapply(10^log.yaxtick[-length(log.yaxtick)], function(i){i*(2:9)}))

fn_bs_profiles = "bs_profiles_LES_4cases_dx10cm.pdf"
pdf(file = fn_bs_profiles, width = 7, height = 2.5)
par(mfrow = c(1,4), mar = c(4,1,0.5,0), oma  = c(0,2.5,0,0.5))#, cex.lab = 1.1, cex.axis = 1.1)

q_bs_LES  = lapply(1:length(my.run1), function(i){ my.run1[[i]]$ptcl$mass.conc / rho_sub })
qn_bs_LES = lapply(1:length(my.run1), function(i){ my.run1[[i]]$ptcl$number.conc / rho_sub })

xlims = range( c( do.call(c, q_bs_top), sapply(q_bs_LES, function(i){ i[1:152] }) ) )
log.xaxtick = seq(floor(log10(xlims[1])), ceiling(log10(xlims[2])), 1)
xaxlab  = sapply(log.xaxtick, function(i){ as.expression(bquote(10^.(i))) })
xaxlab[seq(1,length(xaxlab),2)] = NA
xaxtick.small = do.call(c, lapply(10^log.xaxtick[-length(log.xaxtick)], function(i){i*(2:9)}))
plot(q_bs_LES[[1]], z.plot.LES[[1]], type = "n", log = log.q.plot, xlim = xlims, ylim = ylims, xlab = expression(q[bs]~(kg~kg^{-1})), 
     ylab = "", xaxt = "n", yaxt = "n")
axis(side = 1, at = 10^log.xaxtick, labels = xaxlab)
axis(1, at=xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.15)
axis(2, at= 10^log.yaxtick, labels = yaxlab, las = 1)
axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
title(ylab = "z (m)", line = 2.5, xpd = NA)
for(i in 1:4){
  lines(q_bs_LES[[i]], z.plot.LES[[i]], col = color[i])
}
legend("topright", legend = c("1","2","3a","3b"), title = "Case", col = color, lty = 1, bg = "white")
# add label of subplot
text(x = xlims[1] * (xlims[2]/xlims[1])^0.1, y = 10, labels = "a", cex = 1.3, font = 2)

xlims = range( c( do.call(c, qn_bs_top), sapply(qn_bs_LES, function(i){ i[1:152] }) ) )
log.xaxtick = seq(floor(log10(xlims[1])), ceiling(log10(xlims[2])), 1)
xaxlab  = sapply(log.xaxtick, function(i){ as.expression(bquote(10^.(i))) })
xaxtick.small = do.call(c, lapply(10^log.xaxtick[-length(log.xaxtick)], function(i){i*(2:9)}))
plot(qn_bs_LES[[1]], z.plot.LES[[1]], type = "n", log = log.q.plot, xlim = xlims, ylim = ylims, xlab = expression(N[bs]~(kg^{-1})), 
     ylab = "", xaxt = "n", yaxt = "n")
axis(side = 1, at = 10^log.xaxtick, labels = xaxlab)
axis(1, at=xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.15)
axis(2, at= 10^log.yaxtick, labels = F, las = 1)
axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
for(i in 1:4){
  lines(qn_bs_LES[[i]], z.plot.LES[[i]], col = color[i])
}
# add label of subplot
text(x = xlims[1] * (xlims[2]/xlims[1])^0.1, y = 10, labels = "b", cex = 1.3, font = 2)

xlims = range( sapply(my.run1, function(i){ 1e6*i$ptcl$d_p_mean[1:152] }) )
plot(1e6*my.run1[[1]]$ptcl$d_p_mean, z.plot.LES[[1]], type = "n", log = log.dp.plot, xlim = xlims, ylim = ylims, xlab = expression(bar(d[p])~(mu*m)), ylab = "", yaxt="n")
axis(2, at= 10^log.yaxtick, labels = F, las = 1)
axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
for(i in 1:4){
  lines(1e6*my.run1[[i]]$ptcl$d_p_mean, z.plot.LES[[i]], col = color[i])
}
# add label of subplot
text(x = xlims[1] * (xlims[2]/xlims[1])^0.3, y = 10, labels = "c", cex = 1.3, font = 2)

xlims = range( sapply(my.run1, function(i){ i$ptcl$gamma.shape[1:152] }) )
plot(my.run1[[1]]$ptcl$gamma.shape, z.plot.LES[[1]], type = "n", log = log.dp.plot, xlim = xlims, ylim = ylims, xlab = expression(alpha~(1)), ylab = "", yaxt = "n") #
axis(2, at= 10^log.yaxtick, labels = F, las = 1)
axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
for(i in 1:4){
  lines(my.run1[[i]]$ptcl$gamma.shape, z.plot.LES[[i]], col = color[i])
}
# add label of subplot
text(x = xlims[1] * (xlims[2]/xlims[1])^0.1, y = 10, labels = "d", cex = 1.3, font = 2)
dev.off()


if(write2csv){
  # simulated data
  df.out = list()
  for(i in 1:length(my.run1)){
    df.out[[i]] = data.frame( Case = rep(case.label[i], length(z.w[[i]])),
                        z = c(0,my.run1[[i]]$ta1$z),
                        z_stag = z.w[[i]],
                        u = c(0,my.run1[[i]]$ta1$u),
                        t = c(t0[i], my.run1[[i]]$ta1$t),
                        q = c(q0[i], my.run1[[i]]$ta1$q),
                        RH = c(RH0[i], my.run1[[i]]$ta1$rh),
                        sn_flx = c(NA, round( mass_hflux_les[[i]], digits = 8 ) ),
                        H = H.cum.plot.LES[[i]],
                        LE = LE.cum.plot.LES[[i]],
                        H_MOST = rep(my.run1[[i]]$imitate.MO.meas$qh, length(z.w[[i]])),
                        LE_MOST = rep(my.run1[[i]]$imitate.MO.meas$ql, length(z.w[[i]])) )
    # in.name = match("u",names(df.out))
    # names(df.out)[in.name] = paste(names(df.out)[in.name],"_",case.lab[i],sep="")
    in.flx = which(names(df.out[[i]]) == "sn_flx")
    df.out[[i]][, -c(1,in.flx)] = round(df.out[[i]][, -c(1,in.flx)], digits = 4)
    df.out[[i]]$qbs = c( NA, round( q_bs_LES[[i]], digits = 10 ) )
    df.out[[i]]$Nbs = c( NA, round( qn_bs_LES[[i]] ) )
    df.out[[i]]$dp_mean = c( NA, round( my.run1[[i]]$ptcl$d_p_mean, digits = 6 ) )
    df.out[[i]]$alpha_fit = c( NA, round( my.run1[[i]]$ptcl$gamma.shape, digits = 2 ) )
  }
  df.out = do.call(rbind, df.out)
  write.csv(df.out, file = paste("Fig_2_4_LES.csv",sep=""), na = "NaN", row.names = F)
}


# Plot profiles for set-ups with modeled particle mixing ratios ------------------------------------

ylims = c(0.0075,11.25)
# Show blowing snow profiles with log height axis?
bs_plot_log_z = T
log.q.plot = "x"
log.dp.plot= ""
if(bs_plot_log_z){
  log.q.plot = paste(log.q.plot,"y",sep="")
  log.dp.plot = paste(log.dp.plot,"y",sep="")
  log.yaxtick = seq(floor(log10(ylims[1])), ceiling(log10(ylims[2])), 1)
  yaxlab  = sapply(log.yaxtick, function(i){ as.expression(bquote(10^.(i))) })
  yaxtick.small = do.call(c, lapply(10^log.yaxtick[-length(log.yaxtick)], function(i){i*(2:9)}))
}

# relative x position of subplot label for each case
at_frac = c(0.5, 0.3, 0.45, 0.55)
my.color = c("purple","cyan2","orange","blue",2)
pch.susp = c(5,17,15,20,6) # c(20,15,1,17,6)
# index of model set-ups to be plotted (in my.bs.option)
in.bs.option = c(1,3,6,2,4)
fn_bs_setups = paste("bs_LE_SuspSalt_dx10cm_Case",case.label,".pdf", sep = "")

qi.ABC.list = qni.ABC.list = dp.ABC.list = list()

for(in_case_plot in 1:4){
  my.count = 0
  pdf(file = fn_bs_setups[in_case_plot], width = 7, height = 3.4) #6
  par(mfrow = c(1,4), mar = c(0.7,1,0.5,0), oma  = c(3.3,2.5,4,0.5))
  
  my.count = my.count + 1
  q_bs_LES  = my.run1[[in_case_plot]]$ptcl$mass.conc / rho_sub
  qn_bs_LES = my.run1[[in_case_plot]]$ptcl$number.conc / rho_sub
  
  my.j = which(grepl("transient-Tp", my.option) & (!grepl("1salt-level", my.option)))
  qi_ABC = my.d[[in_case_plot]][[my.j]]$out.qi
  qi.ABC.list[[in_case_plot]] = qi_ABC
  in.lims = which(bs_Z[[in_case_plot]][[my.j]] <= ylims[2])
  xlims = range( c(q_bs_top[[in_case_plot]], qi_ABC[in.lims]), sapply(my.d.susp[[in_case_plot]][in.bs.option], function(x){ x$out.qi[k, ] }) ) #q_bs_LES[1:152])) # 
  log.xaxtick = seq(floor(log10(xlims[1])), ceiling(log10(xlims[2])), 1)
  xaxlab  = sapply(log.xaxtick, function(i){ as.expression(bquote(10^.(i))) })
  xaxlab[seq(1,length(xaxlab),2)] = NA
  xaxtick.small = do.call(c, lapply(10^log.xaxtick[-length(log.xaxtick)], function(i){i*(2:9)}))
  plot(q_bs_LES, z.plot.LES[[in_case_plot]], type = "n", col = "darkgrey", log = log.q.plot, xlim = xlims, ylim = ylims, xlab = "", ylab = "", xaxt = "n", yaxt = "n")
  if(bs_plot_log_z){
    axis(2, at= 10^log.yaxtick, labels = yaxlab, las = 1)
    axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
  } else{
    axis(2)
  }
  axis(side = 1, at = 10^log.xaxtick, labels = xaxlab) #if(my.row == 1){ F }else{ xaxlab })
  axis(1, at=xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.15)
  title(ylab = "z (m)", line = 2.5, xpd = NA)
  abline(h=range(h_salt_susp[[in_case_plot]]), lty=c(3,4), col = c(1,"orange"))
  title(xlab = expression(q[bs]~(kg~kg^{-1})), xpd = NA)
  for(my.index in 1:5){ #c(4,5)){# ((my.row-1)*4+(1:4))
    i = in.bs.option[my.index]
    z_plot = c(bs_Z_susp[[in_case_plot]][[i]], 11.25)
    q_bs_plot = c(my.d.susp[[in_case_plot]][[i]]$out.qi[k, ], q_bs_top[[in_case_plot]])
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[in_case_plot]][i]) ){
      z_plot = c( h_salt_susp[[in_case_plot]][i], z_plot )
      q_bs_plot = c( my.d.susp[[in_case_plot]][[i]]$q_salt, q_bs_plot )
    }
    lines(q_bs_plot, z_plot, type = "o", col = my.color[my.index], pch = pch.susp[my.index])
  }
  points(qi_ABC, bs_Z[[in_case_plot]][[my.j]], pch = 4, cex = 1.5)
  lines(q_bs_LES, z.plot.LES[[in_case_plot]], type = "l", col = "darkgrey", lwd = 1.5)
  # add label of subplot
  text(x = sqrt(xlims[1] * xlims[2]), y = ylims[2] / (ylims[2]/ylims[1])^0.02, labels = c("a","e","i")[my.count], cex = 1.3, font = 2) #xlims[1] * (xlims[2]/xlims[1])^0.08, ylims[2] - 0.03*diff(ylims)
  my.order = c(5,2,1,4,3)
  legend("topleft", legend = expression("LES-LM", Prescribed~q[bs]*": All_improved", SuspSalt*"-"*Existing, SuspSalt*"-"*italic(Res)*"M-"*SemiForward, Susp*"-"*italic(Res)*"M-"*SemiForward, Susp*"-"*italic(Res)*"H-"*Central, Susp*"-"*italic(Res)*"H-"*Central*"-"*italic(h[salt])*H, h[salt]~(Pomeroy~"&"~Male*","~1992), h[salt]*" = "*0.1275~m),#, h[salt]~(low)),
         inset = c(0,-0.24), xpd = NA, ncol = 3, col = c("darkgrey",1,my.color[my.order],1,"orange"), pch = c(NA, 4, pch.susp[my.order], NA, NA), lty = c(1,NA,rep(1,length(my.color)), 3,4), lwd = c(1.5,rep(1,8)))
  # legend("topleft", legend = expression("LES-LSM", u["*0"]*"ABC_ref", NA, italic(h)*"M-"*italic(d)*"LES-Suspension", italic(h)*"M-"*italic(d)*"200-Suspension", NA, italic(h)*"M-"*italic(d)*"200-Transport", italic(h)*"L-"*italic(d)*"200-Transport", "Existing-Transport", h[salt]~(moderate), h[salt]~(low)),#, h[salt]~(low)),
  #        inset = c(0,-0.25), xpd = NA, ncol = 4, col = c("darkgrey",1,NA, my.color[1:2], NA, my.color[3:length(h_salt_susp[[1]])],"darkgrey",1), pch = c(NA, 4, NA, pch.susp[1:2], NA, pch.susp[3:length(h_salt_susp[[1]])], NA, NA), lty = c(1,NA,NA, 1,1,NA, rep(1,length(h_salt_susp[[1]])-2),4,3), lwd = c(1.5,rep(1,11)))
  # legend("topleft", legend = expression("LES-LSM", "1D model with prescribed"~q[bs]~and~bar(d[p]), "1D model with estimated"~q[bs]~and~bar(d[p]), "1D model with existing approach", h[salt]),
  #        inset = c(0.3,-0.25), ncol = 2, xpd = NA, col = c("darkgrey",1, my.color[4:5], 1), pch = c(NA, 4, pch.susp[4:5], NA), lty = c(1,NA, 1,1,3), lwd = c(1.5,rep(1,11))) #
  
  # drifting snow number mixing ratio (kg-1)
  qni_ABC    = my.d[[in_case_plot]][[my.j]]$out.qni 
  qni.ABC.list[[in_case_plot]] = qni_ABC
  xlims = range(c(qni_ABC[in.lims], qn_bs_top[[in_case_plot]], sapply(my.d.susp[[in_case_plot]][in.bs.option], function(x){ x$out.qni[k, -NCOL(x$out.qni)] }))) #  
  log.xaxtick = seq(floor(log10(xlims[1])), ceiling(log10(xlims[2])), 1)
  xaxlab  = sapply(log.xaxtick, function(i){ as.expression(bquote(10^.(i))) })
  xaxtick.small = do.call(c, lapply(10^log.xaxtick[-length(log.xaxtick)], function(i){i*(2:9)}))
  plot(qn_bs_LES, z.plot.LES[[in_case_plot]], type = "n", log = log.q.plot, col = "darkgrey", xlim = xlims, ylim = ylims, xlab = "", ylab = "", yaxt="n", xaxt = "n")
  axis(side = 1, at = 10^log.xaxtick, labels = xaxlab)
  axis(1, at=xaxtick.small, las=1, labels=rep(F,length(xaxtick.small)), tcl = -0.15)
  if(bs_plot_log_z){
    axis(2, at= 10^log.yaxtick, labels = F)
    axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
  } else{
    axis(2, labels = F)
  }
  abline(h=range(h_salt_susp[[in_case_plot]]), lty=c(3,4), col = c(1,"orange"))
  title(xlab = expression(N[bs]~(kg^{-1})), xpd = NA)
  for(my.index in 1:5){ #c(4,5)){#
    i = in.bs.option[my.index]
    z_plot = c(bs_Z_susp[[in_case_plot]][[i]], 11.25)
    qn_bs_plot = c(my.d.susp[[in_case_plot]][[i]]$out.qni[k, ], qn_bs_top[[in_case_plot]])
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[in_case_plot]][i]) ){
      z_plot = c( h_salt_susp[[in_case_plot]][i], z_plot )
      qn_bs_plot = c( my.d.susp[[in_case_plot]][[i]]$qn_salt, qn_bs_plot )
    }
    lines(qn_bs_plot, z_plot, type = "o", col = my.color[my.index], pch = pch.susp[my.index])
  }
  points(qni_ABC, bs_Z[[in_case_plot]][[my.j]], pch = 4, cex = 1.5)
  lines(qn_bs_LES, z.plot.LES[[in_case_plot]], type = "l", col = "darkgrey", lwd = 1.5)
  text(x = sqrt(xlims[1] * xlims[2]), y = ylims[2] / (ylims[2]/ylims[1])^0.02, labels = c("b","f","j")[my.count], cex = 1.3, font = 2)
  
  dp_ABC      = my.d[[in_case_plot]][[my.j]]$meanR * 2
  dp.ABC.list[[in_case_plot]] = dp_ABC
  xlims = 1e6 * range(c(0.0002, 2*my.d.susp[[in_case_plot]][[1]]$meanR, dp_ABC, 2*meanR_ub[[in_case_plot]], my.run1[[in_case_plot]]$ptcl$d_p_mean[1:152]), na.rm = T)
  plot(1e6*my.run1[[in_case_plot]]$ptcl$d_p_mean, z.plot.LES[[in_case_plot]], type = "n", log = log.dp.plot, col="darkgrey", xlim = xlims, ylim = ylims, xlab = "", ylab = "", yaxt="n", xaxt = "n")
  axis(1, labels = T)
  if(bs_plot_log_z){
    axis(2, at= 10^log.yaxtick, labels = F)
    axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
  } else{
    axis(2, labels = F)
  }
  abline(h=range(h_salt_susp[[in_case_plot]]), lty=c(3,4), col = c(1,"orange"))
  title(xlab = expression(bar(d[p])~(mu*m)), xpd = NA)
  for(my.index in 1:5){ #c(4,5)){#
    i = in.bs.option[my.index]
    z_plot = c(bs_Z_susp[[in_case_plot]][[i]], 11.25)
    dp_mean_plot = 2*c( my.d.susp[[in_case_plot]][[i]]$meanR, meanR_ub[[in_case_plot]][i] )
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[in_case_plot]][i]) ){
      z_plot = c( h_salt_susp[[in_case_plot]][i], z_plot )
      dp_mean_plot = c( my.d.susp[[in_case_plot]][[i]]$alpha_salt / my.d.susp[[in_case_plot]][[i]]$lambda_salt, dp_mean_plot )
    }
    lines(1e6*dp_mean_plot, z_plot, type = "o", col = my.color[my.index], pch = pch.susp[my.index])
  }
  points(1e6*dp_ABC, bs_Z[[in_case_plot]][[my.j]], pch = 4, cex = 1.5)
  lines(1e6*my.run1[[in_case_plot]]$ptcl$d_p_mean, z.plot.LES[[in_case_plot]], type = "l", col="darkgrey", lwd = 1.5)
  text(x = mean(c(xlims[1], xlims[2])), y = ylims[2] / (ylims[2]/ylims[1])^0.02, labels = c("c","g","k")[my.count], cex = 1.3, font = 2)
  
  xlims = range(  c( 0, sapply(my.d.susp[[in_case_plot]][in.bs.option], function(j){ j$LE.cum}), my.d[[in_case_plot]][[4]]$LE.cum, LE.cum.plot.LES[[in_case_plot]][1:152]) )# ) )
  if(in_case_plot == 3) xlims[1] = xlims[1] - 0.3
  z_full_ABC = c(0, bs_Z[[in_case_plot]][[my.j]][1:length(bs_dz[[in_case_plot]][[my.j]])] + 0.5*bs_dz[[in_case_plot]][[my.j]], 9)
  plot(LE.cum.plot.LES[[in_case_plot]][-1], z.w[[in_case_plot]][-1], type = "n", col="darkgrey", log = log.dp.plot, xlim = xlims, ylim = ylims, xlab = "", ylab = "", yaxt = "n", xaxt = "n")
  z_surf = ylims[1]*0.65
  axis(1, labels = T)
  if(bs_plot_log_z){
    axis(2, at= 10^log.yaxtick, labels = F)
    axis(2, at=yaxtick.small, las=1, labels=rep(F,length(yaxtick.small)), tcl = -0.2)
  } else{
    axis(2, labels = F)
  }
  abline(h=range(h_salt_susp[[in_case_plot]]), lty=c(3,4), col = c(1,"orange"))
  title(xlab =  expression(LE~(W~m^{-2})), xpd = NA)
  for(my.index in 1:5){ #c(4,5)){#
    i = in.bs.option[my.index]
    z_full_plot = c(0, bs_Z_susp[[in_case_plot]][[i]][1:length(bs_dz_susp[[in_case_plot]][[i]])] + 0.5*bs_dz_susp[[in_case_plot]][[i]], 9)
    LE.cum.plot = my.d.susp[[in_case_plot]][[i]]$LE.cum
    if( grepl(pattern = "no-salt-level", x = path.par.bs[[in_case_plot]][i]) ){
      z_full_plot = c( z_full_plot[1], mean( c( h_salt_susp[[in_case_plot]][i], my.Z.fine[[in_case_plot]][[i]][1] ) ), z_full_plot[-1] )
      LE.cum.plot = c( rep(LE.cum.plot[1], 2), LE.cum.plot[-1] )
    }
    lines(LE.cum.plot[-1], z_full_plot[-1], type = "o", col = my.color[my.index], pch = pch.susp[my.index])
    in_mark = 1:2 #c(1,length(z_full_plot))
    lines(LE.cum.plot[in_mark], c(z_surf , z_full_plot[in_mark[-1]]), type = "o", lty = 2, col = my.color[my.index], pch = pch.susp[my.index], xpd = NA )
  }
  points(my.d[[in_case_plot]][[4]]$LE.cum, c(z_surf, z_full_ABC[-1]), pch = 4, cex = 1.5, xpd = NA)
  in_mark = 1:2#length(z_full_ABC)
  lines(LE.cum.plot.LES[[in_case_plot]][-1], z.w[[in_case_plot]][-1], col="darkgrey", lwd = 1.5)
  lines(LE.cum.plot.LES[[in_case_plot]][in_mark], c(z_surf,z.w[[in_case_plot]][in_mark[-1]]), lty = 2, col="darkgrey", lwd = 1.5, xpd = NA)
  # if(in_case_plot == cases.plot[1]){
  #   if(identical(cases.plot, 2)){
  #     legend("topleft", inset = c(-2,-0.184), legend = expression("LES-LSM", "ABC_ref", "Default", h[salt]~("ABC_ref"), h[salt]~(default)), xpd = NA, ncol = 3,
  #            col = c("darkgrey",1,my.color[length(h_salt_susp[[1]])],1,"darkgrey"), pch = c(NA, 4, pch.susp[length(h_salt_susp[[1]])], NA), lty = c(1,NA,1,3,4), lwd = c(1.5,rep(1,4))) #inset = c(0,-0.28),
  # legend("topleft", inset = c(-2,-0.184), legend = expression("LES-LSM", "ABC_ref", NA, h[salt]~("ABC_ref")), xpd = NA, ncol = 2,
  #        col = c("darkgrey",1,NA,1), pch = c(NA, 4, NA, NA), lty = c(1,NA,NA,3), lwd = c(1.5,rep(1,3))) #inset = c(0,-0.28),
  # }
  # }
  text(x = xlims[1]+at_frac[in_case_plot]*diff(c(xlims[1],xlims[2])), y = ylims[2] / (ylims[2]/ylims[1])^0.02, labels = c("d","h","l")[my.count], cex = 1.3, font = 2)
  dev.off()
}

# write data to file
if(write2csv){
  df.out = list()
  setup.lab = c("all_improved","SuspSalt_Existing","SuspSalt_ResM_SemiForward","Susp_ResM_SemiForward","Susp_ResH_Central","Susp_ResH_Central_hsaltH")
  my.index.write = c(5,2,1,4,3)
  for(i in 1:length(case)){
    z_hsaltH = c(round( bs_Z_susp[[i]][[6]], digits = 4 ), 11.25)
    z_ResM = c(round( bs_Z_susp[[i]][[3]], digits = 4 ), 11.25)
    z_ResM = c( z_ResM, rep(NA, length(z_hsaltH) - length(z_ResM)) )
    z_ResH = c(round( bs_Z_susp[[i]][[2]], digits = 4 ), 11.25)
    z_ResH = c( z_ResH, rep(NA, length(z_hsaltH) - length(z_ResH)) )
    z_Existing = c( h_salt_susp[[i]][4], round( bs_Z_susp[[i]][[4]], digits = 4 ), 11.25 )
    z_Existing = c( z_Existing, rep(NA, length(z_hsaltH) - length(z_Existing)) )
    z_stag_hsaltH = c(0, round( bs_Z_susp[[i]][[6]][1:length(bs_dz_susp[[i]][[6]])] + 0.5*bs_dz_susp[[i]][[6]], digits = 4 ), 9)
    z_stag_ResM = c(0, round( bs_Z_susp[[i]][[3]][1:length(bs_dz_susp[[i]][[3]])] + 0.5*bs_dz_susp[[i]][[3]], digits = 4 ), 9)
    z_stag_ResM = c( z_stag_ResM, rep(NA, length(z_hsaltH) - length(z_stag_ResM)) )
    z_stag_ResH = c(0, round( bs_Z_susp[[i]][[2]][1:length(bs_dz_susp[[i]][[2]])] + 0.5*bs_dz_susp[[i]][[2]], digits = 4 ), 9)
    z_stag_ResH = c( z_stag_ResH, rep(NA, length(z_hsaltH) - length(z_stag_ResH)) )
    z_stag_Existing = round( c(0, mean( c( h_salt_susp[[i]][4], my.Z.fine[[i]][[4]][1] ) ), bs_Z_susp[[i]][[4]][1:length(bs_dz_susp[[i]][[4]])] + 0.5*bs_dz_susp[[i]][[4]], 9), digits = 4 )
    z_stag_Existing = c( z_stag_Existing, rep(NA, length(z_ResM) - length(z_stag_Existing)) )
    qbs_tmp = sapply(0:5, function(j){
      if(j>0) my.j = in.bs.option[my.index.write[j]]
      my.qbs = if(j==0){ qi.ABC.list[[i]] }else{ my.d.susp[[i]][[my.j]]$out.qi[k, ] }
      my.qbs = c(my.qbs, q_bs_top[[i]])
      if(j==1){ my.qbs = c(my.d.susp[[i]][[my.j]]$q_salt, my.qbs) }
      if(length(my.qbs) < length(z_hsaltH)) my.qbs = c(my.qbs, rep(NA, length(z_hsaltH)-length(my.qbs)))
      return(round( my.qbs, digits = 10 ))
    })
    colnames(qbs_tmp) = paste("qbs_", setup.lab, sep = "")
    Nbs_tmp = sapply(0:5, function(j){
      if(j>0) my.j = in.bs.option[my.index.write[j]]
      my.var = if(j==0){ qni.ABC.list[[i]] }else{ my.d.susp[[i]][[my.j]]$out.qni[k, ] }
      my.var = c(my.var, qn_bs_top[[i]])
      if(j==1){ my.var = c(my.d.susp[[i]][[my.j]]$qn_salt, my.var) }
      if(length(my.var) < length(z_hsaltH)) my.var = c(my.var, rep(NA, length(z_hsaltH)-length(my.var)))
      return(round( my.var ))
    })
    colnames(Nbs_tmp) = paste("Nbs_", setup.lab, sep = "")
    dp_tmp = sapply(0:5, function(j){
      if(j>0) my.j = in.bs.option[my.index.write[j]]
      my.var = if(j==0){ dp.ABC.list[[i]] }else{ 2*my.d.susp[[i]][[my.j]]$meanR }
      my.var = c(my.var, 2*meanR_ub[[i]][1])
      if(j==1){ my.var = c(my.d.susp[[i]][[my.j]]$alpha_salt / my.d.susp[[i]][[my.j]]$lambda_salt, my.var) }
      if(length(my.var) < length(z_hsaltH)) my.var = c(my.var, rep(NA, length(z_hsaltH)-length(my.var)))
      return(round( my.var, digits = 6 ))
    })
    colnames(dp_tmp) = paste("dp_", setup.lab, sep = "")
    LE_tmp = sapply(0:5, function(j){
      if(j>0) my.j = in.bs.option[my.index.write[j]]
      my.var = if(j==0){ my.d[[i]][[4]]$LE.cum } else { my.d.susp[[i]][[my.j]]$LE.cum }
      if(j==1){ my.var = c(rep(my.var[1], 2), my.var[-1]) }
      if(length(my.var) < length(z_hsaltH)) my.var = c(my.var, rep(NA, length(z_hsaltH)-length(my.var)))
      return(round( my.var, digits = 2 ))
    })
    colnames(LE_tmp) = paste("LE_", setup.lab, sep = "")
    df.out[[i]] = data.frame(Case = rep(case.label[i], length(z_hsaltH)), z_Existing, z_ResM, z_ResH, z_hsaltH, 
                             z_stag_Existing, z_stag_ResM, z_stag_ResH, z_stag_hsaltH,
                             qbs_tmp, Nbs_tmp, dp_tmp, LE_tmp)
  }
  df.out = do.call(rbind, df.out)
  write.csv(df.out, file = paste("Fig_8_S8_S9_S10_1Dmodel.csv",sep=""), na = "NaN", row.names = F)
}





