# Script for reading and processing data from a Snow Particle Counter (SPC) and from an LES-LM simulation
# to obtain the particle size distributions at given heights

# Armin Sigmund


# Settings ----------------------------------------------------------------
rm(list=ls())
Sys.setenv("TZ"="GMT") # =UTC
# working directory
setwd("~/Documents/PhD_work/my_paper_drafts/paper_parametrization/data_for_EnviDat/Postprocessing_code/")

# directory of SPC data
path = "~/S17_measurements/"
# sub-directories and filenames of SPC data
files = list(mSPC = 'micro-met-station/mSPC/U190111Snow-04-1sec.csv',
             SPC2 = file.path('SPC-at-MRR-site', c('U190111Snow-04-1sec.csv','U190112Snow-04-1sec.csv')))
# time period for average size distribution
tp = list(Case1 = as.POSIXct(c("2019-01-11 00:40","2019-01-11 00:50")),
          Case2 = as.POSIXct(c("2019-01-12 11:00","2019-01-12 11:10")),
          Case3a = as.POSIXct(c("2019-01-11 21:40","2019-01-11 21:50")),
          Case3b = as.POSIXct(c("2019-01-11 21:40","2019-01-11 21:50")))
# borders of 64 particle diameter classes of SPC 
br = c(30,seq(43,253,7),seq(261,324,7),seq(332,381,7),seq(389,431,7),seq(439,474,7),482,NA)
# breaks for histograms
br.spc = c(br[1:(length(br)-1)],500)
br.les = c(0,seq(1,36,7),br[2:(length(br)-1)],seq(489,800,7),10000)

my.cases = c("1","2","3a","3b")
# Select case to  be processed
case      = "1"
in.case   = match(x = case, table = my.cases)
# path of LES-LM results for each of the four cases
path.LES.Rdata = paste("../LES-LM_particles_case_", my.cases, ".Rdata", sep = "")

# particle density
rho.ice   = 918.4 # kg/m^3
# distr.type = "lnorm"
distr.type = "gamma"

library(zoo)
#install.packages("fitdistrplus")
library(fitdistrplus)
# library(dplyr)


# Read SPC data -----------------------------------------------------------

spc = lapply(files, function(x){
  do.call(rbind, lapply(file.path(path, x), read.table, header = T, dec = ".", sep = ",", stringsAsFactors = F, fill = T))
})
# convert to zoo objects
for(i in 1:length(spc)){
  spctime = as.POSIXct(spc[[i]][,3], format = "%Y/%m/%d %H:%M:%S")
  spc[[i]]= zoo(spc[[i]][,6:ncol(spc[[i]])], spctime)
  # remove duplicated times
  in.dup   = which(duplicated(time(spc[[i]])))
  if(length(in.dup) > 0){spc[[i]] = spc[[i]][-in.dup]}
  print(unique(diff(time(spc[[i]]))))
}

# Average size distribution for given time interval
in.spc = if(in.case==1){ 1:2 }else{ 2 }
# select data
spc.sel = lapply(spc[in.spc], function(x){window(x, start = tp[[in.case]][1], end = tp[[in.case]][2])})
# sum up particle numbers per size bin
spc.sum = lapply(spc.sel, FUN = apply, MARGIN = 2, sum, na.rm = T)
# probability density
spc_dens_particles = lapply(spc.sum, function(x){
  x / ( sum(x) * diff(br.spc) )
})
# create dataframe for distribution fitting with fitdistcens()
censdat = data.frame(left = br[-length(br)], right= br[-1])

x.mids.spc = (br.spc[-length(br.spc)] + br.spc[-1]) / 2
# sample mean
d_p_avg = lapply(spc.sum, function(x){ sum(x * x.mids.spc) / sum(x) })
# sample variance
var_d_p = lapply(1:length(spc.sum), function(k){ sum( spc.sum[[k]] * (x.mids.spc - d_p_avg[[k]])^2 ) / sum( spc.sum[[k]] ) })
if(distr.type == "gamma"){
  my.start = lapply(1:length(d_p_avg), function(k){ 
    list(shape=d_p_avg[[k]]^2/var_d_p[[k]], rate=d_p_avg[[k]]/var_d_p[[k]])
  })
} else{
  stop(paste("distr.type",distr.type,"not fully implemented."))
}
fit.distr = lapply(1:length(spc.sum), function(k){
  fitdistcens(censdat, weights = spc.sum[[k]], keepdata = F, distr = distr.type, start = my.start[[k]]) # weights must be strictly positive integers)
})
lapply(fit.distr, summary)


# LES particle size distributions -----------------------------------------------------------------------------

# Load particles data
load(path.LES.Rdata[in.case])
# number of snow particles per Lagrangian particle
ptcl$ppp = ptcl$mass_p / (rho.ice*pi*ptcl$d_p^3 / 6)

# LES: average size distribution at given heights
# select parcels with a height of approximately 0.15 and 0.10 m, respectively
height = if(in.case == 1){ c(0.15, 0.1) }else{ 0.1 }
in.z = lapply(height, function(x){which(abs(ptcl$z_p - x) <= 0.0025)})
# index of diameter bin
ptcl$d_p_bin = findInterval(ptcl$d_p * 1e+6, br.les)
# probability density ({mu}m-1)
les_dens_particles = lapply(in.z, function(x){
  # number of particles per diameter bin
  n_tmp = aggregate(ptcl$ppp[x], list( ptcl$d_p_bin[x] ), sum)
  n_particles = rep(0, length(br.les) - 1)
  n_particles[n_tmp[, 1]] = n_tmp[, 2]
  # probability density
  n_particles / ( sum(n_particles) * diff(br.les) )
})

x.mids.les = (br.les[-length(br.les)] + br.les[-1]) / 2
# sample mean
d_p_avg_les = lapply(les_dens_particles, function(x){ 
  my_n = x * diff(br.les)
  sum(my_n * x.mids.les)
})
# sample variance
var_d_p_les = lapply(1:length(d_p_avg_les), function(k){ 
  my_n = les_dens_particles[[k]] * diff(br.les)
  sum( my_n * (x.mids.les - d_p_avg_les[[k]])^2 )
  })
if(distr.type == "gamma"){
  my.start = lapply(1:length(d_p_avg_les), function(k){ 
    list(shape=d_p_avg_les[[k]]^2/var_d_p_les[[k]], rate=d_p_avg_les[[k]]/var_d_p_les[[k]])
  })
} else{
  stop(paste("distr.type",distr.type,"not fully implemented."))
}
# convert to micrometer because fitting algorithm does not work if values are very small
d_p_i = lapply(in.z, function(x){ ptcl$d_p[x] * 1e6 })
ppp_i = lapply(in.z, function(x){ round(ptcl$ppp[x]) })
fit.les.distr = lapply(1:length(in.z), function(k){ fitdist(d_p_i[[k]], distr = "gamma", keepdata = F, weights = ppp_i[[k]], start = my.start[[k]]) })
lapply(fit.les.distr, summary)

names(d_p_i) = height #c("z15","z10")

fit.les.distr = lapply(d_p_i, fitdist, distr = distr.type, keepdata = F)


# save data which is needed for size distribution plot ------------------------------------------------------
save(spc_dens_particles, fit.distr, les_dens_particles, fit.les.distr, height, file = paste("PSD_data_for_plot_case_", my.cases[in.case], ".Rdata", sep = ""))

