# Script for calculating and plotting cospectra from high-frequency (e.g. 20 Hz) time series from LES output and field data

# Armin Sigmund

# Settings ----------------------------------------------------------------
rm(list=ls())
Sys.setenv("TZ"="GMT") # =UTC

library(zoo)
library(oce)

setwd("~/Documents/PhD_work/my_paper_drafts/paper_parametrization/data_for_EnviDat/Postprocessing_code/")

case   = "1"
prefix = paste("Case_", case, "_", sep = "")

if(case == "1"){
  
  path         = "~/Documents/PhD_work/LES_output_postprocessing/snowdrift_antarctica/S17_190111_0045/deep_domain/case1_dx10cm/"
  path.eddypro = "../Turbulence_measurements_case_1.txt"
  # output file name for plot
  fn.co.plot = "cospectra_wh2o_wT_LES-case1_vs_field_detrend"
  
} else{ if(case == "2"){
  
  path         = "~/Documents/PhD_work/LES_output_postprocessing/snowdrift_antarctica/S17_190112_1105/case2_dx10cm/"

} else{ if(case == "3a"){
  
  path         = "~/Documents/PhD_work/LES_output_postprocessing/snowdrift_antarctica/S17_190111_2145/case3a_dx10cm/"

} else{ if(case == "3b"){
  
  path         = "~/Documents/PhD_work/LES_output_postprocessing/snowdrift_antarctica/S17_190111_2145/case3b_dx10cm/"

} else{
            warning("The specified case is not known.")
}}}}

dt.eddy.cov = 0.05

t.start  = 0
# estimated beginning of quasi-stationary T and q conditions (s)
t.start.stnry = 350
# simulation parameters
rho_air = 1.18 # kg/m^3
# molar mass of water
Mw = 0.01802 # kg mol^-1

# number of records for input into FFT (power of 2 is fastest)
N = 2^11*3
# averaging window for detrending the time series using a running mean (number of records)
win = 3001

# taping function of Hamming: function of number of data points to the center of the time series (abs(l) from 0 to N/2)
# hamming = function(l, N){ 0.54 + 0.46 * cos(2*pi*l/N) }
# exponential function for spacing of frequency bins
calc.f.bins = function(x){ exp(1)^(0.1745 * x - 6.499) } # almost like in EddyPro
#calc.f.bins = function(x){ exp(1)^(0.04795347 * x) - 0.9975586 }

# color vector
cv = c("black","deepskyblue","red","orange","darkgrey")
# define line widths
lwd.small  = 0.8
lwd.normal = 1
lwd.fat    = 1.2

# Should LES single-point output be saved as csv file (for publication in Envidat repository)?
write2csv = T


# read high-frequency data (grid point at certain level in center of LES domain) -----------------------

dat = read.table(file = file.path(path, "output_0", "eddy_cov", "uwTq.txt"), header = T, sep = "")

# convert to zoo with time in sec
time = seq(t.start + dt.eddy.cov, by = dt.eddy.cov, length.out = NROW(dat))
dat    = zoo(dat, time)

## store data for Envidat
if(write2csv){
  d.out = cbind(time(dat), as.data.frame(dat))
  colnames(d.out) = c("Time","u","w","T","q")
  # unit conversion
  d.out$T = d.out$T - 273.15
  d.out$q = d.out$q * 1000
  d.out[,2:NCOL(d.out)] = round(d.out[,2:NCOL(d.out)], digits = 6)
  fn.out = paste(prefix, "uwTq_single_point.csv")
  write.csv(d.out, fn.out, row.names = F)
}


# calculate water vapor molar density (mmol m^-3) from specific humidity to have the same units as in field measurements such that (co)variances can be compared
dat$h2o = dat$q.kg.kg. * rho_air / Mw * 1000
dat$q.kg.kg. = NULL

# reduce time series to a power of two records (2^13 = 8192) plus half of the running mean window on both sides 
d    = dat[(NROW(dat)-(N+win-1)+1) : NROW(dat), ] # use records towards the end of the quasi-stationary period

# read measured high-freq data (eddyPro output) ----------------------------
d = list(LES = d)
d$field = read.table(file = path.eddypro, skip = 9, header = T, sep = "")
# replace -9999 by NA
d$field[which(d$field == -9999, arr.ind = T)] = NA
time.field = seq(dt.eddy.cov, by = dt.eddy.cov, length.out = NROW(d$field))
d$field = zoo(d$field, time.field)
# reduce time series to a power of 2 records or similar (e.g., 2^13 = 8192) plus half of the running mean window on both sides 
d$field = d$field[seq(800, by = 1, length.out = N+win-1), ]

# linearly interpolate gaps up to 3 consecutive points
d$field = do.call(cbind, lapply(1:NCOL(d$field), function(x){
  na.approx(object = d$field[,x], x = time(d$field[,x]), maxgap = 3)
}))
names(d$field) = names(d$LES)
# Are there still data gaps in the field data?
any(is.na(d$field))


# calculate (co)spectra --------------------------------------------------------

stat = list()
stat.detrend = list()
run.mean = list()
d.detrend = list()
S.all = list()
og.all= list()
S.bin = list()
for(k in 1:2){
  # (1) (co)variances of original time series with a length of power of 2 records (2^13 = 8192 records)
  in.use = seq(ceiling(0.5*win), by = 1, length.out = N)
  stat[[k]] = data.frame(var.w = var(d[[k]]$W.m.s.[in.use], na.rm = T),
                    var.T = var(d[[k]]$T.K.[in.use], na.rm = T),
                    var.h2o = var(d[[k]]$h2o[in.use], na.rm = T),
                    cov.wT  = cov(x = d[[k]]$W.m.s.[in.use], y = d[[k]]$T.K.[in.use], use = "pairwise.complete.obs"),
                    cov.wh2o= cov(x = d[[k]]$W.m.s.[in.use], y = d[[k]]$h2o[in.use], use = "pairwise.complete.obs"))
  
  # (2a) remove trend in time series using running mean because LES time series contain artificial trend due to locked large-scale coherent structures 
  run.mean[[k]] = rollapply(data = d[[k]], width = win, by = 1, by.column = T, align = "left", partial = F, FUN = mean)
  # shift time to convert from left edge to center of runinng mean window
  time(run.mean[[k]]) = time(run.mean[[k]]) + dt.eddy.cov * floor(0.5*win)
  # remove trend (need to use coredata matrix instead of zoo object to get correct number of rows)
  d.detrend[[k]] = as.data.frame( coredata(d[[k]][in.use, ]) - coredata(run.mean[[k]]) )
  
  # (2b) (co)variances of detrended time series
  stat.detrend[[k]] = data.frame(var.w = var(d.detrend[[k]]$W.m.s., na.rm = T),
                                 var.T = var(d.detrend[[k]]$T.K., na.rm = T),
                                 var.h2o = var(d.detrend[[k]]$h2o, na.rm = T),
                                 cov.wT  = cov(x = d.detrend[[k]]$W.m.s., y = d.detrend[[k]]$T.K., use = "pairwise.complete.obs"),
                                 cov.wh2o= cov(x = d.detrend[[k]]$W.m.s., y = d.detrend[[k]]$h2o, use = "pairwise.complete.obs"))
  
  # (3) tapering with Hamming window
  #f.hamming = hamming(-2^12:(2^12-1), 2^13)
  # tapered data: fluctuation times tapering function
  # cnames = names(d.detrend[[k]])
  # d.detrend[[k]]   = do.call(cbind, lapply(1:NCOL(d.detrend[[k]]), function(i){
    # tmp = d.detrend[[k]][,i] - mean(d.detrend[[k]][,i])
    #tmp * f.hamming
  # }))
  # colnames(d.detrend[[k]]) = cnames
  
  # (4) FFT and spectral density
  in.proc = 2:4
  xfft = sapply(in.proc, function(i){
    fft( ts(d.detrend[[k]][,i], start = 0, frequency = 20) ) / N 
  }) # divide by N because this is missing in the formula specified in the documentation, otherwise integral of power spectrum not equal to variance
  colnames(xfft) = colnames(d.detrend[[k]])[in.proc]
  # Info from book of Stull: First element reflects the mean value of the time series (frequency of zero), will be excluded
  # Summing the square of the norm of the complex coefficients (= coefficient times its complex conjugate) gives the variance of the original time series
  # Second element is complex conjugate of last element, third element is complex conjugate of second last element, ...
  # Thus, all information is included in the range from second element to (N/2+1)-th element, corresponding to frequencies of 1/time.period to 1/(2*delta.t), i.e. Nyquist frequency.
  # frequencies (Hz)
  delta.f = 1/(dt.eddy.cov*N)
  f = seq( delta.f, 1/(2*dt.eddy.cov), delta.f )
  # Spectral density for power spectrum (if w was input of fft, unit of spectral density is m^2 s^-2 Hz^-1)
  S.power = sapply(1:NCOL(xfft), function(i){ 
    tmp = Re( xfft[2:(N/2+1), i] * Conj(xfft[2:(N/2+1), i]) ) / delta.f
    # Multiply with 2 except for Nyquist frequency
    tmp[1:(length(tmp)-1)] = 2 * tmp[1:(length(tmp)-1)]
    return(tmp)
  })
  colnames(S.power) = colnames(d.detrend[[k]])[in.proc]
  # Check whether integral is equal to variance
  sum(S.power[,2]*delta.f)
  
  # Spectral density for cospectrum (units for wT cospectrum: m s^-1 K Hz^-1)
  in.co = list(wT = 1:2, wh2o = c(1,3))
  S.co = do.call(cbind, lapply(in.co, function(i){
    j = 2:(N/2+1)
    tmp = ( Re(xfft[j, i[1]]) * Re(xfft[j, i[2]]) + Im(xfft[j, i[1]]) * Im(xfft[j, i[2]]) ) / delta.f
    # Multiply with 2 except for Nyquist frequency
    tmp[1:(length(tmp)-1)] = 2 * tmp[1:(length(tmp)-1)]  
    return(tmp)
  }))
  colnames(S.co) = names(in.co)
  # Check whether integral is equal to covariance
  sum(S.co[,2]*delta.f)
  
  S.all[[k]] = cbind(S.power, S.co)
  
  # normalized spectral density: divide by integral of (co)spectrum and multiply with frequency
  # spec.norm = lapply(1:length(spec.density), function(x){
  #   spec.density[[x]]$spec / integral.spec[[x]] * spec.density[[x]]$freq
  # })
  
  
  # ogives ------------------------------------------------------------------
  og.all[[k]] = do.call(cbind, lapply(1:NCOL(S.all[[k]]), function(x){
    tmp = rev(cumsum(rev(S.all[[k]][,x] * delta.f)))
    zoo(tmp, f)
  }))
  colnames(og.all[[k]]) = colnames(S.all[[k]])
  
  
  # bin averaging -----------------------------------------------------------
  # define 51 borders (for 50 frequency bins)
  f.br = calc.f.bins(seq(2.7725, 50.439, length.out = 51))
  # bin average
  S.bin[[k]] = do.call(cbind, lapply(1:NCOL(S.all[[k]]), function(i){
    tmp = binMean1D(x = f, f = S.all[[k]][,i], xbreaks = f.br)
    zoo(tmp$result, tmp$xmids)
  }))
  colnames(S.bin[[k]]) = colnames(S.all[[k]])
  # Check integral
  sum(S.bin[[k]]$wh2o * diff(f.br), na.rm = T)
}
names(stat) = names(S.all) = names(og.all) = names(S.bin) = names(d)


# Plot cospectrum and ogive ---------------------------------------------------------
# Figure S2

# legend text
leg.lty = c(1,1,2,1,1)
leg.col = cv[c(1,2,5,3,4)]
leg.pch = c(20,17,NA,NA,NA)
xmin  = f[1]   # plot limits
xmax  = tail(f, n=1)    
xlabs = expression(italic(f)~(Hz))
xlims = range(index(S.bin$LES$wh2o))
# avoid space at the left and right borders of plotting region
xlims = exp( log( xlims ) + c(0.08, -0.05) * log( diff(xlims) ) )
# log10 of y coordinate of starting points for line representing Kolmogorov cospectra
log.y.kolmo = c(-1.5, -2.5) # S = 10^log.y.kolmo

pdf(paste(fn.co.plot, ".pdf", sep = ""), width=11, height=4.5)
# cairo_ps(filename = paste(fn.co.plot, ".eps", sep = ""), width = 6, height = 4.5, onefile = T)
# png(paste(fn.co.plot, ".png", sep = ""), width = 22, height = 9, pointsize = 24, units="in", res=150)
par(mar=c(4.1,4.4,1.5,4.1), mfrow = c(1,2))

for(j in 1:2){ # j=1 refers to wh2o, j=2 to wT
  if(j == 1){
    # cospectra list to be plotted, omit NA's
    xy    = list(LES   = na.omit(S.bin$LES$wh2o), 
                 field = na.omit(S.bin$field$wh2o))
    xy.og = list(LES   = og.all$LES$wh2o,
                 field = og.all$field$wh2o)
    # axis labels
    ylabs = expression(italic(S[wH2O])~(mmol~m^{-2}))
    ylabs2= expression(italic(Og[wH2O])~(mmol~m^{-2}~s^{-1}))
    leg.txt = expression(italic(S)*","~simulation, italic(S)*","~field, "-7/3"~slope, italic(Og)*","~simulation, italic(Og)*","~field)
  } else{ # wT
    xy    = list(LES   = -1*na.omit(S.bin$LES$wT), 
                 field = -1*na.omit(S.bin$field$wT))
    xy.og = list(LES   = -1*og.all$LES$wT,
                 field = -1*og.all$field$wT)
    ylabs = expression(-italic(S[wT])~(m~K))
    ylabs2= expression(-italic(Og[wT])~(m~s^{-1}~K))
    leg.txt = expression(-italic(S)*","~simulation, -italic(S)*","~field, "-7/3"~slope, -italic(Og)*","~simulation, -italic(Og)*","~field)
  }
  
  tmp   = do.call(cbind, xy)
  tmp   = apply(tmp, 2, function(i){
    i[which(i < 0)] = NA
    return(i)
  })
  ymin  = min(tmp, na.rm = T) # 1e-3  # plot limits
  ymax  = max(tmp, na.rm = T) + 4 # 1e+2
  ylim2 = range(do.call(cbind, xy.og)) #stat$field$cov.wh2o
  # axis ticks
  xaxlab  = 10^seq(floor(log10(xmin)), ceiling(log10(xmax)), 1)
  xaxtick = do.call(c, lapply(xaxlab[-length(xaxlab)], function(i){i*(2:9)}))
  yaxlab  = 10^seq(floor(log10(ymin)), ceiling(log10(ymax)), 1)
  yaxtick = do.call(c, lapply(yaxlab[-length(yaxlab)], function(i){i*(2:9)}))
  
  count = 0
  for( i in 1:length(xy) ){
    count = count+1
    if(count == 1){
      # log-log plot of the cospectrum as function of frequency
      plot(x = index(xy[[i]]), y = coredata(xy[[i]]), type = "o", pch = 20, cex = 0.7, log = "xy", col=cv[count], 
           xlim = xlims, ylim = c(ymin,ymax), lwd = lwd.normal, xlab = xlabs, ylab = ylabs, xaxt = "n", yaxt = "n")
      # line for Kolmogorov cospectrum: slope of -7/3 for cospectrum
      lines(x = 10^c(0.813, 0.813+3), y = 10^c(log.y.kolmo[j], log.y.kolmo[j]-7), lty=2, col=cv[5], lwd = lwd.normal)
      # vertical line at time scale corresponding to spatial scale of 6 m
      # abline(v = f.highlight, lty=2) #, lwd=lwd.normal)
      legend("topright", legend=leg.txt, lty = leg.lty, pch = leg.pch, col = leg.col, bg = "white", x.intersp = 0.3) 
      axis(1, at= xaxlab,  las=1, labels=xaxlab)
      axis(1, at= xaxtick, las=1, labels=rep(F,length(xaxtick)), tcl = -0.3)
      axis(2, at= yaxlab,  las=1, labels=yaxlab)
      axis(2, at= yaxtick, las=1, labels=rep(F,length(yaxtick)), tcl = -0.3)
    } else{
      lines(x = index(xy[[i]]), y = coredata(xy[[i]]), type = "o", pch = 17, cex = 0.7, lwd = lwd.normal, col = cv[count])
    }
  }
  ## add ogive on second y axis
  # Allow a second plot on the same graph
  par(new=TRUE)
  #count = count + 1
  plot(x = index(xy.og$LES), y = coredata(xy.og$LES), type = "l", log = "x", xlim = xlims, ylim = ylim2, 
       col = cv[3], axes = F, xlab="", ylab="")
  lines(x = index(xy.og$field), y = coredata(xy.og$field), col = cv[4])
  axis(4, ylim=ylim2, col="red",col.axis="red",las=1)
  mtext(ylabs2,side=4,col="red",line=3)
  # add label of subplot
  text(x = xlims[1], ylim2[2] - 0.05*(ylim2[2]-ylim2[1]), labels = letters[j], cex = 1.5, font = 2)
}

dev.off()




