# Functions for calculating turbulent heat fluxes and vertical profiles according to the Monin-Obukhov bulk formulation. 

# author: Armin Sigmund


# General functions -------------------------------------------------------

# flux in kinematic units
calc_bulk_kinemat_flux = function(u.up, u.low, scalar.up, scalar.low, mycoeff){
  -1 * mycoeff * (u.up - u.low) * (scalar.up - scalar.low)
}

# bulk transfer coefficient for scalars
calc_coeff_scalar = function(psi.m, psi.s, z.u.up, z.u.low, z.scalar.up, z.scalar.low){
  0.4^2/((log(z.u.up/z.u.low)-psi.m)*(log(z.scalar.up/z.scalar.low)-psi.s))
}


# Stable stratification ---------------------------------------------------

# Universal functions for stable and neutral conditions according to Stearns and Weidner (1993).
## zeta is supposed to be >= 0
calc_psi_stable_stearns_weidner = function(zeta){
  if(any(zeta < 0)) stop("zeta is not >= 0")
  # momentum
  y   = (1+5*zeta)^0.25
  psi = data.frame(m = log((1+y)^2) + log(1+y^2) - 2*atan(y) - 4/3 * y^3 + 0.8247)
  # temperature, scalars
  y      = (1+5*zeta)^0.5
  psi$s  = log((1+y)^2) - 2*y - 2/3 * y^3 + 1.2804
  return(psi)
}

# Universal functions for stable and neutral conditions according to Holtslag and DeBruin (1988).
## zeta is supposed to be >= 0
calc_psi_stable_holtslag = function(zeta){
  if(any(zeta < 0)) stop("zeta is not >= 0")
  # same for momentum and scalars
  psi    = data.frame(m = -(0.7 * zeta + 0.75 * (zeta - 14.28) * exp(-0.35 * zeta) + 10.71))
  psi$s  = psi$m
  return(psi)
}


# Unstable stratification -------------------------------------------------

# universal functions after Businger et al., 1971 using the normalization after Hoegstroem, 1988
calc_psi_unstable = function(zeta){
  x.term = (1 - 19.3*zeta)^0.25
  data.frame(m  = log( ( (1+x.term^2)/2 ) * ( (1 + x.term)/2 )^2 ) - 2*atan(x.term) + pi / 2 ,
             s  = 2 * log( (1 + (0.95*(1 - 11.6*zeta)^0.5)^2 ) / 2 ))
}

# universal functions of Paulson (original) for momentum and of Stearns and Weidner (1993) for scalars
# [The Stearns and Weidner (1993) formula for momentum is not correct, something (at least a sign of one term) seems to be missing in the paper.]
# zeta is supposed to be < 0
calc_psi_unstable_paulson_stearns_weidner = function(zeta){
  if(any(zeta >= 0)) stop("zeta is not < 0")
  # momentum
  x   = (1-15*zeta)^0.25
  psi = data.frame( m = 2 * log(0.5 * (1+x)) + log(0.5 * (1+x^2)) - 2 * atan(x) + 0.5 * pi )
  # temperature, scalars
  x      = (1-22.5*zeta)^(1/3)
  psi$s  = log((1+x+x^2)^1.5) - 3^0.5 * atan((1 + 2*x)*3^(-0.5)) + 0.1659 
  # Here, the paper is not clear about whether to first apply the power of 1.5 and then the logarithm or first the logarithm and the the power of 1.5; however, the former makes more sense because otherwise psi$s does not approach zero but -0.496 for zeta approaching zero.
  return(psi)
}


# Iterative flux calculation -------------------------------------------------
# z_ref_vw        reference height (m) for vw_ref, only used if prescribe_ustar = NA
# z_ref_scalar    reference height (m) for T_ref and qv_ref
# rough_len_m     roughness length for momentum (m), only used if prescribe_ustar = NA
# rough_len_tq_Andreas:   logical indicating whether to calculate scalar roughness lengths using the parametrization of Andreas (1987). If TRUE, the file func_z0_ratio_Andreas.R needs to be sourced.
# rough_len_t     roughness length for  temperature (m), only used if rough_len_tq_Andreas is FALSE
# rough_len_q     roughness length for  humidity (m)   , only used if rough_len_tq_Andreas is FALSE
# vw_ref          wind speed at ref height (m s-1), only used if prescribe_ustar = NA
# T_ref           temperature at ref height (K to be consistent with T_surf)
# qv_ref          specific humidity at ref height (kg kg-1)
# T_surf          surface temperature (K)
# qv_surf         surface specific humidity (kg kg-1)
# FUN_psi_stable  universal function for stable conditions
# FUN_psi_unstable  universal function for unstable conditions
# prescribe_ustar   either NA (vw_ref and rough_len_m are used) or numeric value of a prescribed friction velocity

# output:    Data frame with friction velocity (u.star, m s-1), temperature flux (Tw_flux, K m s-1), moisture flux (qw_flux, kg kg-1 m s-1), 
#            stability parameter (zeta, dimensionless), integrated universal functions for momentum (psi.m, dimensionless) and scalars (psi.s, dimensionless),
#            and optionally the roughness Reynolds number (if rough_len_tq_Andreas is TRUE)

calc_fluxes_iter = function(z_ref_vw = z_ref_scalar, z_ref_scalar = z_ref_vw, rough_len_m, rough_len_tq_Andreas = F, rough_len_t = 0.1*rough_len_m, rough_len_q = 0.1*rough_len_m, vw_ref, T_ref, qv_ref, T_surf, qv_surf, FUN_psi_stable = calc_psi_stable_holtslag, FUN_psi_unstable = calc_psi_unstable_paulson_stearns_weidner, prescribe_ustar = NA){
  # constants
  nu.air = 1.24e-05
  # if wind speed is zero, set fluxes to zero and skip iterative calculation
  iterate = T
  if(is.na(prescribe_ustar)){
    if(vw_ref == 0){
      u.star = 0
      Tw_flux = 0
      qw_flux = 0
      zeta = NA
      psi = list(m = NA, s  = NA)
      Re.star = NA
      iterate = F
    }
  }
  # iterative flux calculation
  if(iterate){
    for (i in 1:100){
      if (i == 1){
        # start with neutral conditions
        zeta = 1e-6 # not 0 because at the end of the loop it is divided by zeta
      } else{
        zeta = zeta.new
      }
      # stable or neutral conditions
      if(zeta >= 0){
        # limit zeta to 10
        if(zeta > 10) zeta = 10
        # integrated form of universal functions after Holtslag and deBruin
        psi = FUN_psi_stable(zeta = zeta)
      } else{ # unstable conditions
        # limit zeta to -10
        if(zeta < -10) zeta = -10
        # parameterization after Paulson (for momentum) and after Stearns and Weidner (for scalars) 
        psi = FUN_psi_unstable(zeta)
      }
      # friction velocity (m s^-1)
      if(is.finite(prescribe_ustar)){
        u.star = prescribe_ustar
      } else{
        u.star = 0.4 * vw_ref / ( log(z_ref_vw / rough_len_m) - psi$m )
      }
      # scalar roughness lengths according to Andreas parametrization
      if(rough_len_tq_Andreas){
        Re.star = u.star * rough_len_m / nu.air
        # roughness length for temperature
        rough_len_t = rough_len_m * calc.Andreas.model(R.star = Re.star, "temperature")      
        # roughness length for moisture
        rough_len_q = rough_len_m * calc.Andreas.model(R.star = Re.star, "humidity")
      }
      if(is.finite(prescribe_ustar)){
        Tw_flux = -0.4 * u.star * (T_ref - T_surf) / ( log(z_ref_scalar/rough_len_t) - psi$s)
      } else{
        # bulk transfer coefficient for temperature
        c.h = calc_coeff_scalar(psi.m = psi$m, psi.s = psi$s, z.u.up = z_ref_vw, z.u.low = rough_len_m, z.scalar.up = z_ref_scalar, z.scalar.low = rough_len_t)
        # sensible heat flux in kinematic units (K m s-1)
        Tw_flux = calc_bulk_kinemat_flux(u.up = vw_ref, u.low = 0, scalar.up = T_ref, scalar.low = T_surf, mycoeff = c.h)
      }
      # stability parameter z/L
      zeta.new = - z_ref_scalar * 0.4 * (9.81/T_surf) * Tw_flux / u.star^3
      if ( abs((zeta.new - zeta) / zeta) < 0.001 ) break
    }
    if(is.finite(prescribe_ustar)){
      qw_flux = -0.4 * u.star * (qv_ref - qv_surf) / ( log(z_ref_scalar/rough_len_q) - psi$s)
    } else{
      # bulk transfer coefficient for humidity
      c.q = calc_coeff_scalar(psi.m = psi$m, psi.s = psi$s, z.u.up = z_ref_vw, z.u.low = rough_len_m, z.scalar.up = z_ref_scalar, z.scalar.low = rough_len_q)
      # latent heat flux in kinematic units (kg kg-1 m s-1)
      qw_flux = calc_bulk_kinemat_flux(u.up = vw_ref, u.low = 0, scalar.up = qv_ref, scalar.low = qv_surf, mycoeff = c.q)
    }
  }
  out = data.frame(u.star, Tw_flux, qw_flux, zeta, psi$m, psi$s)
  if(rough_len_tq_Andreas) out = cbind(out, Re.star)
  return(out)
}


# Vertical profiles ---------------------------------------------------------------------------
# z_ref      reference height (m)
# x_ref      quantity of interest at ref height
# x_star     vertical flux of quantity of interest divided by friction velocity (unit of the quantity of interest)
# psi        integrated universal function, stability correction term
# z_out      output heights (m)

calc_MO_profile = function(z_ref, x_ref, x_star, psi, z_out){
  x_ref  + x_star / 0.4 * ( log(z_out/z_ref) - psi )
}

