# Function for calculating a vertical profile from particle data by averaging in each layer
# data            vector to be aggregated, each element refers to a parcel
# INDEX           list of two vectors, namely time and layer identifier, each with the same length as data, used in tapply
# ncol            number of columns (layers) to be allocated
# ind.column      index of layers covered by data
# normaliz.matrix matrix used for normalization of aggregated data, result is divided by normaliz.matrix[ , ind.column], e.g. total mass per time and layer
# default         default value for initialization of aggregated data, this value is used for layers without parcels
# with.time       logical indicating whether a time-dependent profile should additonally be outputed

# OUTPUT: Either a vector representing the time-averaged and layer-averaged vertical profile
#         or a list containing the same vector and a matrix with the time-dependent vertical profile

extract_profile_from_particles = function(data, INDEX, ncol, ind.column, normaliz.matrix = NULL, default = NA, with.time = F){

  # Sum of quantity per layer and point in time
  var.tmp = tapply(X = data, INDEX = INDEX, FUN = sum, na.rm = T, default = default)
  if(!is.null(normaliz.matrix)) var.tmp = var.tmp / normaliz.matrix[ , ind.column]
  if(length(ind.column) != NCOL(var.tmp)) stop("Length of ind.column is not equal to number of columns in the aggregated data matrix.")
  if(with.time){
    # mass flux as a function of time
    var_with_time               = matrix(default, nrow = NROW(var.tmp), ncol = ncol)
    var_with_time[,ind.column]  = var.tmp
  }
  # average over time
  var.tmp = apply( var.tmp, 2, mean, na.rm = T )
  var     = rep(default, ncol)
  var[ind.column] = var.tmp
  if(with.time){
    return( list(var, var_with_time) )
  } else{
    return( var )
  }
}
